<?php
session_start();
require_once '../../api/db.php';
require_once '../../api/persian_date.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

// Disable cache
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

// Get filter parameters
$period = $_GET['period'] ?? 'month';
$start_date = $_GET['start_date'] ?? null;
$end_date = $_GET['end_date'] ?? null;

// Calculate date range
$now = time();
switch ($period) {
    case 'today':
        $start = strtotime('today');
        $end = $now;
        break;
    case 'week':
        $start = strtotime('-7 days');
        $end = $now;
        break;
    case 'month':
        $start = strtotime('-30 days');
        $end = $now;
        break;
    case 'custom':
        $start = $start_date ? strtotime($start_date) : strtotime('-30 days');
        $end = $end_date ? strtotime($end_date) : $now;
        break;
    default:
        $start = strtotime('-30 days');
        $end = $now;
}

// Get orders in range
try {
    $stmt = $conn->prepare("SELECT * FROM orders WHERE created_at >= ? AND created_at <= ? ORDER BY created_at DESC");
    $stmt->execute([$start * 1000, $end * 1000]);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $orders = [];
}

// Calculate statistics
$stats = [
    'total_orders' => count($orders),
    'total_income' => 0,
    'completed' => 0,
    'pending' => 0,
    'cancelled' => 0,
    'avg_order' => 0
];

$daily_income = [];
foreach ($orders as $order) {
    if ($order['status'] != 'cancelled') {
        $stats['total_income'] += $order['total_price'];
    }
    if ($order['status'] == 'delivered') $stats['completed']++;
    if ($order['status'] == 'pending') $stats['pending']++;
    if ($order['status'] == 'cancelled') $stats['cancelled']++;
    
    $day = date('Y-m-d', $order['created_at'] / 1000);
    if (!isset($daily_income[$day])) $daily_income[$day] = 0;
    if ($order['status'] != 'cancelled') {
        $daily_income[$day] += $order['total_price'];
    }
}

if ($stats['total_orders'] > 0) {
    $stats['avg_order'] = $stats['total_income'] / ($stats['total_orders'] - $stats['cancelled']);
}

// Export to Excel
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="report_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // UTF-8 BOM
    
    fputcsv($output, ['کد پیگیری', 'نام مشتری', 'شماره تماس', 'مبلغ', 'وضعیت', 'تاریخ']);
    
    foreach ($orders as $order) {
        fputcsv($output, [
            $order['tracking_code'],
            $order['customer_name'],
            $order['customer_phone'],
            $order['total_price'],
            $order['status'],
            getPersianDate('Y/m/d H:i', $order['created_at'] / 1000)
        ]);
    }
    
    fclose($output);
    exit();
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>گزارش‌گیری و آمار | پاکشو</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@100..900&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body { font-family: 'Vazirmatn', sans-serif; }
        .glass-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(229, 231, 235, 0.5);
        }
    </style>
</head>
<body class="bg-slate-50 text-gray-800 antialiased">
    <div class="min-h-screen flex flex-col">
        
        <!-- Header -->
        <header class="bg-white/90 backdrop-blur border-b border-gray-200 sticky top-0 z-50 px-6 py-4">
            <div class="max-w-7xl mx-auto flex items-center justify-between">
                <div class="flex items-center gap-5">
                    <a href="settings.php" class="w-10 h-10 rounded-xl bg-gray-50 border border-gray-200 hover:bg-white hover:border-blue-300 hover:text-blue-600 hover:shadow-md transition-all duration-300 flex items-center justify-center text-gray-500">
                        <i class="fa-solid fa-arrow-right"></i>
                    </a>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">گزارش عملکرد</h1>
                        <p class="text-xs text-gray-500 mt-0.5">آنالیز دقیق فروش و وضعیت سفارشات</p>
                    </div>
                </div>
                
                <a href="?period=<?php echo $period; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>&export=excel" 
                   class="flex items-center gap-2 px-5 py-2.5 bg-emerald-600 hover:bg-emerald-700 text-white font-bold rounded-xl shadow-lg shadow-emerald-200 transition-all transform hover:-translate-y-0.5">
                    <i class="fa-solid fa-file-excel"></i>
                    <span class="hidden sm:inline">خروجی اکسل</span>
                </a>
            </div>
        </header>

        <main class="flex-1 p-6 md:p-8">
            <div class="max-w-7xl mx-auto space-y-8">
                
                <!-- Filter Section -->
                <div class="bg-white rounded-2xl border border-gray-200 p-2 shadow-sm">
                    <form method="GET" class="flex flex-col sm:flex-row items-center gap-2 w-full">
                        <div class="flex-1 w-full sm:w-auto p-2">
                            <label class="block text-xs font-bold text-gray-500 mb-1.5 mr-1">بازه نمایش</label>
                            <div class="relative">
                                <select name="period" onchange="this.form.submit()" class="w-full pl-4 pr-10 py-2.5 bg-gray-50 border border-gray-200 rounded-lg focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all text-sm appearance-none cursor-pointer">
                                    <option value="today" <?php echo $period == 'today' ? 'selected' : ''; ?>>امروز</option>
                                    <option value="week" <?php echo $period == 'week' ? 'selected' : ''; ?>>هفته جاری</option>
                                    <option value="month" <?php echo $period == 'month' ? 'selected' : ''; ?>>یک ماه اخیر</option>
                                    <option value="custom" <?php echo $period == 'custom' ? 'selected' : ''; ?>>بازه سفارشی</option>
                                </select>
                                <span class="absolute left-3 top-3 text-gray-500 pointer-events-none"><i class="fa-solid fa-calendar-days text-xs"></i></span>
                            </div>
                        </div>

                        <?php if ($period == 'custom'): ?>
                        <div class="flex flex-col sm:flex-row gap-2 w-full sm:w-auto flex-[2] p-2 bg-gray-50 rounded-xl border border-dashed border-gray-300">
                            <div class="w-full">
                                <label class="block text-xs font-bold text-gray-500 mb-1.5 mr-1">از تاریخ</label>
                                <input type="date" name="start_date" value="<?php echo $start_date; ?>" class="w-full px-3 py-2 border border-gray-200 rounded-lg text-sm">
                            </div>
                            <div class="w-full">
                                <label class="block text-xs font-bold text-gray-500 mb-1.5 mr-1">تا تاریخ</label>
                                <input type="date" name="end_date" value="<?php echo $end_date; ?>" class="w-full px-3 py-2 border border-gray-200 rounded-lg text-sm">
                            </div>
                            <div class="flex items-end">
                                <button type="submit" class="w-full sm:w-auto px-6 py-2 bg-blue-600 text-white rounded-lg text-sm font-bold hover:bg-blue-700 transition">فیلتر</button>
                            </div>
                        </div>
                        <?php endif; ?>
                    </form>
                </div>

                <!-- Stats Cards -->
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
                    <!-- Total Income -->
                    <div class="glass-card p-6 rounded-3xl relative overflow-hidden group">
                        <div class="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                            <i class="fa-solid fa-coins text-8xl text-emerald-600 transform rotate-12 translate-x-4 -translate-y-4"></i>
                        </div>
                        <div class="relative z-10">
                            <div class="w-12 h-12 rounded-2xl bg-emerald-100 text-emerald-600 flex items-center justify-center text-xl mb-4 shadow-sm">
                                <i class="fa-solid fa-sack-dollar"></i>
                            </div>
                            <span class="block text-sm font-medium text-gray-500 mb-1">درآمد کل</span>
                            <div class="flex items-baseline gap-1">
                                <span class="text-3xl font-black text-gray-800 tracking-tight"><?php echo number_format($stats['total_income']); ?></span>
                                <span class="text-xs font-bold text-gray-400">تومان</span>
                            </div>
                        </div>
                    </div>

                    <!-- Total Orders -->
                    <div class="glass-card p-6 rounded-3xl relative overflow-hidden group">
                        <div class="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                            <i class="fa-solid fa-receipt text-8xl text-blue-600 transform rotate-12 translate-x-4 -translate-y-4"></i>
                        </div>
                        <div class="relative z-10">
                            <div class="w-12 h-12 rounded-2xl bg-blue-100 text-blue-600 flex items-center justify-center text-xl mb-4 shadow-sm">
                                <i class="fa-solid fa-cart-shopping"></i>
                            </div>
                            <span class="block text-sm font-medium text-gray-500 mb-1">تعداد سفارشات</span>
                            <div class="flex items-baseline gap-1">
                                <span class="text-3xl font-black text-gray-800 tracking-tight"><?php echo number_format($stats['total_orders']); ?></span>
                                <span class="text-xs font-bold text-gray-400">عدد</span>
                            </div>
                        </div>
                    </div>

                    <!-- Avg Order -->
                    <div class="glass-card p-6 rounded-3xl relative overflow-hidden group">
                        <div class="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                            <i class="fa-solid fa-chart-line text-8xl text-purple-600 transform rotate-12 translate-x-4 -translate-y-4"></i>
                        </div>
                        <div class="relative z-10">
                            <div class="w-12 h-12 rounded-2xl bg-purple-100 text-purple-600 flex items-center justify-center text-xl mb-4 shadow-sm">
                                <i class="fa-solid fa-calculator"></i>
                            </div>
                            <span class="block text-sm font-medium text-gray-500 mb-1">میانگین هر سفارش</span>
                            <div class="flex items-baseline gap-1">
                                <span class="text-3xl font-black text-gray-800 tracking-tight"><?php echo number_format($stats['avg_order']); ?></span>
                                <span class="text-xs font-bold text-gray-400">تومان</span>
                            </div>
                        </div>
                    </div>

                    <!-- Success Rate -->
                    <div class="glass-card p-6 rounded-3xl relative overflow-hidden group">
                        <div class="absolute top-0 right-0 p-4 opacity-10 group-hover:opacity-20 transition-opacity">
                            <i class="fa-solid fa-check-double text-8xl text-indigo-600 transform rotate-12 translate-x-4 -translate-y-4"></i>
                        </div>
                        <div class="relative z-10">
                            <div class="w-12 h-12 rounded-2xl bg-indigo-100 text-indigo-600 flex items-center justify-center text-xl mb-4 shadow-sm">
                                <i class="fa-solid fa-clipboard-check"></i>
                            </div>
                            <span class="block text-sm font-medium text-gray-500 mb-1">سفارشات تکمیل شده</span>
                            <div class="flex items-baseline gap-1">
                                <span class="text-3xl font-black text-gray-800 tracking-tight"><?php echo number_format($stats['completed']); ?></span>
                                <span class="text-xs font-bold text-gray-400">موفق</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Main Content Grid -->
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    
                    <!-- Chart Section (Takes 2 columns) -->
                    <div class="lg:col-span-2 bg-white rounded-3xl border border-gray-200 p-6 shadow-sm">
                        <div class="flex items-center justify-between mb-6">
                            <h2 class="text-lg font-bold text-gray-800">روند درآمد</h2>
                            <span class="text-xs font-medium bg-gray-100 text-gray-500 px-3 py-1 rounded-full">بر اساس تاریخ</span>
                        </div>
                        <div class="relative h-80 w-full">
                            <canvas id="incomeChart"></canvas>
                        </div>
                    </div>

                    <!-- Summary / Mini Stats (Takes 1 column) -->
                    <div class="lg:col-span-1 space-y-6">
                         <!-- Order Status Breakdown -->
                        <div class="bg-white rounded-3xl border border-gray-200 p-6 shadow-sm h-full">
                            <h2 class="text-lg font-bold text-gray-800 mb-6">وضعیت سفارشات</h2>
                            <div class="space-y-4">
                                <div class="flex items-center justify-between p-3 rounded-xl bg-amber-50 border border-amber-100">
                                    <div class="flex items-center gap-3">
                                        <div class="w-2 h-2 rounded-full bg-amber-500"></div>
                                        <span class="text-sm font-bold text-gray-700">در انتظار</span>
                                    </div>
                                    <span class="font-bold text-amber-700"><?php echo $stats['pending']; ?></span>
                                </div>
                                <div class="flex items-center justify-between p-3 rounded-xl bg-emerald-50 border border-emerald-100">
                                    <div class="flex items-center gap-3">
                                        <div class="w-2 h-2 rounded-full bg-emerald-500"></div>
                                        <span class="text-sm font-bold text-gray-700">تکمیل شده</span>
                                    </div>
                                    <span class="font-bold text-emerald-700"><?php echo $stats['completed']; ?></span>
                                </div>
                                <div class="flex items-center justify-between p-3 rounded-xl bg-rose-50 border border-rose-100">
                                    <div class="flex items-center gap-3">
                                        <div class="w-2 h-2 rounded-full bg-rose-500"></div>
                                        <span class="text-sm font-bold text-gray-700">لغو شده</span>
                                    </div>
                                    <span class="font-bold text-rose-700"><?php echo $stats['cancelled']; ?></span>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>

                <!-- Orders Table -->
                <div class="bg-white rounded-3xl border border-gray-200 shadow-sm overflow-hidden">
                    <div class="px-6 py-5 border-b border-gray-100 bg-gray-50/50 flex items-center justify-between">
                        <h2 class="font-bold text-gray-800">ریز تراکنش‌ها</h2>
                        <span class="text-xs text-gray-500"><?php echo count($orders); ?> مورد یافت شد</span>
                    </div>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50 text-gray-500 text-xs uppercase font-bold tracking-wider">
                                <tr>
                                    <th class="px-6 py-4 text-right">کد پیگیری</th>
                                    <th class="px-6 py-4 text-right">مشخصات مشتری</th>
                                    <th class="px-6 py-4 text-center">مبلغ (تومان)</th>
                                    <th class="px-6 py-4 text-center">وضعیت</th>
                                    <th class="px-6 py-4 text-left">تاریخ ثبت</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-100">
                                <?php if(empty($orders)): ?>
                                <tr>
                                    <td colspan="5" class="px-6 py-12 text-center text-gray-400">
                                        <i class="fa-solid fa-inbox text-4xl mb-3 opacity-50"></i>
                                        <p>هیچ سفارشی در این بازه یافت نشد</p>
                                    </td>
                                </tr>
                                <?php endif; ?>

                                <?php foreach ($orders as $order): ?>
                                <tr class="hover:bg-blue-50/30 transition-colors group">
                                    <td class="px-6 py-4">
                                        <span class="font-mono font-bold text-gray-700 bg-gray-100 px-2 py-1 rounded text-sm group-hover:bg-white border border-transparent group-hover:border-gray-200 transition-all">
                                            #<?php echo htmlspecialchars($order['tracking_code']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="flex flex-col">
                                            <span class="font-bold text-gray-800 text-sm mb-1"><?php echo htmlspecialchars($order['customer_name']); ?></span>
                                            <span class="text-xs text-gray-500 font-mono"><?php echo htmlspecialchars($order['customer_phone']); ?></span>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-center">
                                        <span class="font-black text-gray-800 text-base"><?php echo number_format($order['total_price']); ?></span>
                                    </td>
                                    <td class="px-6 py-4 text-center">
                                        <?php
                                        $status_config = [
                                            'pending' => ['bg' => 'bg-amber-100', 'text' => 'text-amber-700', 'label' => 'در انتظار', 'icon' => 'fa-clock'],
                                            'processing' => ['bg' => 'bg-blue-100', 'text' => 'text-blue-700', 'label' => 'شستشو', 'icon' => 'fa-soap'],
                                            'ready' => ['bg' => 'bg-indigo-100', 'text' => 'text-indigo-700', 'label' => 'آماده', 'icon' => 'fa-check'],
                                            'delivered' => ['bg' => 'bg-emerald-100', 'text' => 'text-emerald-700', 'label' => 'تحویل شده', 'icon' => 'fa-check-double'],
                                            'cancelled' => ['bg' => 'bg-rose-100', 'text' => 'text-rose-700', 'label' => 'لغو شده', 'icon' => 'fa-xmark']
                                        ];
                                        $status = $status_config[$order['status']] ?? $status_config['pending'];
                                        ?>
                                        <span class="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full text-xs font-bold <?php echo $status['bg'] . ' ' . $status['text']; ?>">
                                            <i class="fa-solid <?php echo $status['icon']; ?>"></i>
                                            <?php echo $status['label']; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-left">
                                        <div class="text-sm font-medium text-gray-600 dir-ltr text-right">
                                            <?php echo getPersianDate('Y/m/d', $order['created_at'] / 1000); ?>
                                            <span class="text-xs text-gray-400 block mt-0.5"><?php echo getPersianDate('H:i', $order['created_at'] / 1000); ?></span>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>

<?php
$persian_labels = [];
$sorted_dates = array_keys($daily_income);
sort($sorted_dates);
foreach ($sorted_dates as $date) {
    $timestamp = strtotime($date);
    $persian_labels[] = getPersianDate('Y/m/d', $timestamp);
}
$sorted_income = [];
foreach ($sorted_dates as $date) {
    $sorted_income[] = $daily_income[$date];
}
?>
    <script>
        const ctx = document.getElementById('incomeChart').getContext('2d');
        const persianLabels = <?php echo json_encode($persian_labels); ?>;
        const incomeData = <?php echo json_encode($sorted_income); ?>;
        
        // Create gradient
        const gradient = ctx.createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, 'rgba(16, 185, 129, 0.4)');   
        gradient.addColorStop(1, 'rgba(16, 185, 129, 0.0)');

        // Initialize chart
        const chart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: persianLabels,
                datasets: [{
                    label: 'درآمد (تومان)',
                    data: incomeData,
                    borderColor: '#059669',
                    backgroundColor: gradient,
                    borderWidth: 3,
                    pointBackgroundColor: '#fff',
                    pointBorderColor: '#059669',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    pointHoverRadius: 6,
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.8)',
                        titleFont: { family: 'Vazirmatn' },
                        bodyFont: { family: 'Vazirmatn' },
                        padding: 10,
                        cornerRadius: 8,
                        callbacks: {
                            label: function(context) {
                                return new Intl.NumberFormat('fa-IR').format(context.raw) + ' تومان';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            font: { family: 'Vazirmatn' },
                            callback: function(value) {
                                return new Intl.NumberFormat('fa-IR', { notation: 'compact' }).format(value);
                            }
                        }
                    },
                    x: {
                        grid: { display: false },
                        ticks: { font: { family: 'Vazirmatn' } }
                    }
                }
            }
        });
        
        // Auto-refresh data every 5 seconds via AJAX
        setInterval(async () => {
            try {
                const params = new URLSearchParams(window.location.search);
                const res = await fetch('../../api/reports_data.php?' + params.toString() + '&t=' + Date.now());
                const data = await res.json();
                
                if (data.stats && data.chart) {
                    // Update stats cards
                    document.querySelectorAll('.glass-card .text-3xl')[0].textContent = new Intl.NumberFormat('fa-IR').format(data.stats.total_income);
                    document.querySelectorAll('.glass-card .text-3xl')[1].textContent = new Intl.NumberFormat('fa-IR').format(data.stats.total_orders);
                    document.querySelectorAll('.glass-card .text-3xl')[2].textContent = new Intl.NumberFormat('fa-IR').format(data.stats.avg_order);
                    document.querySelectorAll('.glass-card .text-3xl')[3].textContent = new Intl.NumberFormat('fa-IR').format(data.stats.completed);
                    
                    // Update status breakdown
                    document.querySelectorAll('.bg-amber-50 .font-bold.text-amber-700')[0].textContent = data.stats.pending;
                    document.querySelectorAll('.bg-emerald-50 .font-bold.text-emerald-700')[0].textContent = data.stats.completed;
                    document.querySelectorAll('.bg-rose-50 .font-bold.text-rose-700')[0].textContent = data.stats.cancelled;
                    
                    // Update chart
                    chart.data.labels = data.chart.labels;
                    chart.data.datasets[0].data = data.chart.data;
                    chart.update('none');
                    
                    // Update orders table
                    if (data.orders) {
                        const tbody = document.querySelector('table tbody');
                        if (data.orders.length === 0) {
                            tbody.innerHTML = '<tr><td colspan="5" class="px-6 py-12 text-center text-gray-400"><i class="fa-solid fa-inbox text-4xl mb-3 opacity-50"></i><p>هیچ سفارشی در این بازه یافت نشد</p></td></tr>';
                        } else {
                            tbody.innerHTML = data.orders.map(order => {
                                const statusConfig = {
                                    'pending': {bg: 'bg-amber-100', text: 'text-amber-700', label: 'در انتظار', icon: 'fa-clock'},
                                    'processing': {bg: 'bg-blue-100', text: 'text-blue-700', label: 'شستشو', icon: 'fa-soap'},
                                    'ready': {bg: 'bg-indigo-100', text: 'text-indigo-700', label: 'آماده', icon: 'fa-check'},
                                    'delivered': {bg: 'bg-emerald-100', text: 'text-emerald-700', label: 'تحویل شده', icon: 'fa-check-double'},
                                    'cancelled': {bg: 'bg-rose-100', text: 'text-rose-700', label: 'لغو شده', icon: 'fa-xmark'}
                                };
                                const status = statusConfig[order.status] || statusConfig['pending'];
                                return `<tr class="hover:bg-blue-50/30 transition-colors group">
                                    <td class="px-6 py-4"><span class="font-mono font-bold text-gray-700 bg-gray-100 px-2 py-1 rounded text-sm group-hover:bg-white border border-transparent group-hover:border-gray-200 transition-all">#${order.tracking_code}</span></td>
                                    <td class="px-6 py-4"><div class="flex flex-col"><span class="font-bold text-gray-800 text-sm mb-1">${order.customer_name}</span><span class="text-xs text-gray-500 font-mono">${order.customer_phone}</span></div></td>
                                    <td class="px-6 py-4 text-center"><span class="font-black text-gray-800 text-base">${new Intl.NumberFormat('fa-IR').format(order.total_price)}</span></td>
                                    <td class="px-6 py-4 text-center"><span class="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full text-xs font-bold ${status.bg} ${status.text}"><i class="fa-solid ${status.icon}"></i>${status.label}</span></td>
                                    <td class="px-6 py-4 text-left"><div class="text-sm font-medium text-gray-600 dir-ltr text-right">${order.created_at_persian}<span class="text-xs text-gray-400 block mt-0.5">${order.created_time_persian}</span></div></td>
                                </tr>`;
                            }).join('');
                        }
                    }
                }
            } catch (e) {
                console.error('Failed to update reports:', e);
            }
        }, 5000);
    </script>
</body>
</html>