<?php
session_start();
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");
require_once __DIR__ . '/../../api/db.php';
require_once __DIR__ . '/../../api/persian_date.php';
require_once __DIR__ . '/../../includes/system.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

$licenseCheck = System::check();
$licenseValid = $licenseCheck['v'] == 1;

try {
    $orders = $conn->query("SELECT * FROM orders ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
    
    // Get unique customers from orders
    $customers_data = [];
    foreach($orders as $order) {
        $key = $order['customer_phone'];
        if (!isset($customers_data[$key])) {
            $customers_data[$key] = [
                'name' => $order['customer_name'],
                'phone' => $order['customer_phone'],
                'address' => $order['customer_address'] ?? '',
                'orders_count' => 0,
                'total_spent' => 0
            ];
        }
        $customers_data[$key]['orders_count']++;
        if ($order['status'] != 'cancelled') {
            $customers_data[$key]['total_spent'] += $order['total_price'];
        }
    }
    $customers = array_values($customers_data);
    
    $stats = ['total' => count($orders), 'pending' => 0, 'income' => 0, 'today' => 0];
    $monthly_income = array_fill(0, 12, 0);
    $month_names = ['فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'];
    $today_start = strtotime('today') * 1000;
    
    foreach($orders as &$o) {
        $ts = $o['created_at'] > 10000000000 ? $o['created_at'] / 1000 : $o['created_at'];
        $o['created_at_persian'] = getPersianDate('Y/m/d', $ts);
        
        if (!empty($o['delivery_date'])) {
            if (is_numeric($o['delivery_date'])) {
                $delivery_ts = $o['delivery_date'] > 10000000000 ? $o['delivery_date'] / 1000 : $o['delivery_date'];
                $o['delivery_date_persian'] = getPersianDate('Y/m/d', $delivery_ts);
            } else {
                $o['delivery_date_persian'] = $o['delivery_date'];
            }
        } else {
            $o['delivery_date_persian'] = null;
        }
        
        if ($o['status'] == 'pending') $stats['pending']++;
        if ($o['status'] != 'cancelled') {
            $stats['income'] += $o['total_price'];
            $order_month = (int)date('n', $ts);
            $order_year = (int)date('Y', $ts);
            $current_year = (int)date('Y');
            if ($order_year == $current_year || $order_year == $current_year - 1) { 
                $monthly_income[$order_month - 1] += $o['total_price'];
            }
        }
        if ($o['created_at'] >= $today_start) $stats['today']++;
    }
    
    $max_income = max($monthly_income);
    $chart_data = [];
    for($i = 0; $i < 12; $i++) {
        $chart_data[] = [
            'month' => $month_names[$i],
            'amount' => $monthly_income[$i],
            'height' => $max_income > 0 ? round(($monthly_income[$i] / $max_income) * 100) : 0
        ];
    }
} catch (PDOException $e) {
    $orders = [];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0">
    <title>داشبورد مدیریت | پاکشو</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@100..900&display=swap" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: { sans: ['Vazirmatn', 'sans-serif'] },
                    colors: {
                        primary: { 50: '#eff6ff', 100: '#dbeafe', 500: '#3b82f6', 600: '#2563eb', 700: '#1d4ed8' }
                    }
                }
            }
        }
    </script>
    <style>
        [x-cloak] { display: none !important; }
        ::-webkit-scrollbar { width: 6px; height: 6px; }
        ::-webkit-scrollbar-track { background: transparent; }
        ::-webkit-scrollbar-thumb { background: #cbd5e1; border-radius: 10px; }
    </style>
</head>
<body class="bg-gray-50 text-gray-800 font-sans antialiased h-screen flex overflow-hidden" x-data="adminDashboard()">

    <!-- Toast Notifications -->
    <div class="fixed top-6 left-1/2 -translate-x-1/2 z-[100] flex flex-col gap-2 pointer-events-none">
        <template x-for="toast in toasts" :key="toast.id">
            <div x-show="toast.visible" x-transition class="pointer-events-auto flex items-center gap-3 px-4 py-3 rounded-xl shadow-lg border backdrop-blur-md min-w-[320px]"
                 :class="toast.type === 'success' ? 'bg-emerald-500/90 text-white border-emerald-400' : 'bg-rose-500/90 text-white border-rose-400'">
                <i :class="toast.type === 'success' ? 'fa-circle-check' : 'fa-circle-exclamation'"></i>
                <span x-text="toast.message" class="text-sm font-medium"></span>
            </div>
        </template>
    </div>

    <!-- Mobile Sidebar Overlay -->
    <div x-show="sidebarOpen" @click="sidebarOpen = false" x-transition.opacity class="fixed inset-0 bg-gray-900/50 z-20 lg:hidden backdrop-blur-sm"></div>

    <!-- Sidebar -->
    <aside :class="sidebarOpen ? 'translate-x-0' : 'translate-x-full lg:translate-x-0'" class="fixed lg:static inset-y-0 right-0 w-72 bg-white border-l border-gray-200 flex flex-col z-30 shadow-lg transition-transform duration-300">
        <div class="h-20 flex items-center px-8 border-b border-gray-100/50">
            <div class="flex items-center gap-3 text-primary-600">
                <div class="w-10 h-10 rounded-xl bg-primary-50 flex items-center justify-center text-xl shadow-sm">
                    <i class="fa-solid fa-soap"></i>
                </div>
                <span class="font-black text-2xl tracking-tight text-gray-900">پاکشو</span>
            </div>
        </div>
        <nav class="flex-1 p-6 space-y-2 overflow-y-auto">
            <a href="../../activate.php" class="flex items-center gap-3 px-4 py-3.5 rounded-2xl font-medium transition-all <?= !$licenseValid ? 'bg-amber-50 text-amber-700 ring-1 ring-amber-200' : 'text-gray-500 hover:text-gray-900 hover:bg-gray-50' ?>">
                <i class="fa-solid fa-key w-6"></i> فعالسازی
                <?php if (!$licenseValid): ?>
                <span class="mr-auto bg-red-500 text-white text-xs px-2 py-0.5 rounded-full">!</span>
                <?php endif; ?>
            </a>
            <a href="#" class="flex items-center gap-3 px-4 py-3.5 rounded-2xl font-bold shadow-sm ring-1 transition-all <?= $licenseValid ? 'bg-primary-50 text-primary-700 ring-primary-100' : 'bg-gray-100 text-gray-400 cursor-not-allowed' ?>" <?= !$licenseValid ? 'onclick="return false;"' : '' ?>>
                <i class="fa-solid fa-chart-pie w-6"></i> داشبورد
            </a>
            <a href="order_create.php" class="flex items-center gap-3 px-4 py-3.5 rounded-2xl font-medium transition-all <?= $licenseValid ? 'text-gray-500 hover:text-gray-900 hover:bg-gray-50' : 'text-gray-400 cursor-not-allowed' ?>" <?= !$licenseValid ? 'onclick="return false;"' : '' ?>>
                <i class="fa-solid fa-plus-circle w-6"></i> سفارش جدید
            </a>
            <a href="customers.php" class="flex items-center gap-3 px-4 py-3.5 rounded-2xl font-medium transition-all <?= $licenseValid ? 'text-gray-500 hover:text-gray-900 hover:bg-gray-50' : 'text-gray-400 cursor-not-allowed' ?>" <?= !$licenseValid ? 'onclick="return false;"' : '' ?>>
                <i class="fa-solid fa-users w-6"></i> لیست مشتریان
            </a>
            <a href="settings.php" class="flex items-center gap-3 px-4 py-3.5 rounded-2xl font-medium transition-all <?= $licenseValid ? 'text-gray-500 hover:text-gray-900 hover:bg-gray-50' : 'text-gray-400 cursor-not-allowed' ?>" <?= !$licenseValid ? 'onclick="return false;"' : '' ?>>
                <i class="fa-solid fa-gear w-6"></i> تنظیمات
            </a>
        </nav>
        <div class="p-6 border-t border-gray-100">
            <div class="flex items-center gap-3 p-3 rounded-2xl bg-gray-50 border border-gray-100">
                <img src="https://ui-avatars.com/api/?name=<?php echo urlencode($_SESSION['full_name'] ?? 'Admin'); ?>&background=2563eb&color=fff" class="w-10 h-10 rounded-xl">
                <div class="flex-1 min-w-0">
                    <div class="text-sm font-bold text-gray-900 truncate"><?php echo $_SESSION['full_name'] ?? 'مدیر سیستم'; ?></div>
                    <div class="text-xs text-gray-400">مدیریت کل</div>
                </div>
                <a href="../../logout.php" class="w-8 h-8 flex items-center justify-center rounded-lg text-gray-400 hover:bg-rose-50 hover:text-rose-600 transition-colors">
                    <i class="fa-solid fa-power-off"></i>
                </a>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 flex flex-col min-w-0 bg-gray-50/50 relative">
        <header class="h-20 px-4 md:px-8 bg-white/80 backdrop-blur-xl border-b border-gray-200 flex items-center justify-between sticky top-0 z-10">
            <div class="flex items-center gap-4">
                <button @click="sidebarOpen = !sidebarOpen" class="lg:hidden p-2 text-gray-500 hover:bg-gray-100 rounded-lg"><i class="fa-solid fa-bars text-xl"></i></button>
                <div>
                    <h1 class="text-xl font-bold text-gray-800">داشبورد</h1>
                    <p class="text-xs text-gray-400 hidden sm:block mt-0.5">خلاصه وضعیت خشکشویی</p>
                </div>
            </div>
            <div class="flex items-center gap-3 md:gap-4">
                <div class="relative hidden sm:block group">
                    <i class="fa-solid fa-search absolute right-3.5 top-1/2 -translate-y-1/2 text-gray-400"></i>
                    <input type="text" x-model="search" placeholder="جستجو (نام، موبایل)..." class="w-64 h-11 pr-10 pl-4 bg-gray-100 border-transparent focus:bg-white focus:border-primary-500 border-2 rounded-xl text-sm transition-all outline-none">
                </div>
                <a href="order_create.php" class="h-11 px-6 bg-gray-900 hover:bg-gray-800 text-white text-sm font-bold rounded-xl flex items-center gap-2 transition-all shadow-lg active:scale-95">
                    <i class="fa-solid fa-plus"></i> <span class="hidden sm:inline">ثبت سفارش</span>
                </a>
            </div>
        </header>

        <div class="flex-1 overflow-y-auto p-4 md:p-8">
            <div class="max-w-7xl mx-auto space-y-6">
                <!-- Stats -->
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 md:gap-6">
                    <div class="bg-white p-6 rounded-3xl border border-gray-100 shadow-sm hover:shadow-lg transition-shadow">
                        <div class="flex justify-between items-start mb-4">
                            <div class="w-12 h-12 rounded-2xl bg-blue-50 text-blue-600 flex items-center justify-center text-xl">
                                <i class="fa-solid fa-calendar-day"></i>
                            </div>
                            <span class="px-2.5 py-1 rounded-full bg-gray-100 text-gray-500 text-xs font-bold">امروز</span>
                        </div>
                        <div class="text-3xl font-black text-gray-900 mb-1"><?php echo number_format($stats['today']); ?></div>
                        <p class="text-sm text-gray-500 font-medium">سفارش جدید</p>
                    </div>
                    <div class="bg-white p-6 rounded-3xl border border-gray-100 shadow-sm hover:shadow-lg transition-shadow">
                        <div class="flex justify-between items-start mb-4">
                            <div class="w-12 h-12 rounded-2xl bg-amber-50 text-amber-600 flex items-center justify-center text-xl">
                                <i class="fa-solid fa-clock"></i>
                            </div>
                            <span class="px-2.5 py-1 rounded-full bg-amber-100 text-amber-700 text-xs font-bold">اقدام</span>
                        </div>
                        <div class="text-3xl font-black text-gray-900 mb-1"><?php echo number_format($stats['pending']); ?></div>
                        <p class="text-sm text-gray-500 font-medium">در انتظار</p>
                    </div>
                    <div class="bg-white p-6 rounded-3xl border border-gray-100 shadow-sm hover:shadow-lg transition-shadow">
                        <div class="flex justify-between items-start mb-4">
                            <div class="w-12 h-12 rounded-2xl bg-emerald-50 text-emerald-600 flex items-center justify-center text-xl">
                                <i class="fa-solid fa-sack-dollar"></i>
                            </div>
                            <span class="px-2.5 py-1 rounded-full bg-emerald-100 text-emerald-700 text-xs font-bold">درآمد</span>
                        </div>
                        <div class="text-3xl font-black text-gray-900 mb-1"><?php echo number_format($stats['income']); ?></div>
                        <p class="text-sm text-gray-500 font-medium">تومان</p>
                    </div>
                    <div class="bg-white p-6 rounded-3xl border border-gray-100 shadow-sm hover:shadow-lg transition-shadow">
                        <div class="flex justify-between items-start mb-4">
                            <div class="w-12 h-12 rounded-2xl bg-purple-50 text-purple-600 flex items-center justify-center text-xl">
                                <i class="fa-solid fa-box-open"></i>
                            </div>
                        </div>
                        <div class="text-3xl font-black text-gray-900 mb-1"><?php echo number_format($stats['total']); ?></div>
                        <p class="text-sm text-gray-500 font-medium">کل سفارشات</p>
                    </div>
                </div>

                <!-- Table -->
                <div class="bg-white border border-gray-100 rounded-3xl shadow-sm overflow-hidden">
                    <div class="p-6 border-b border-gray-100">
                        <div class="flex items-center gap-2 mb-4">
                            <button @click="activeTab = 'orders'" class="px-4 py-2 rounded-xl font-bold text-sm transition-all" :class="activeTab === 'orders' ? 'bg-primary-600 text-white' : 'text-gray-500 hover:bg-gray-100'">
                                <i class="fa-solid fa-box ml-2"></i>سفارشات
                            </button>
                            <button @click="activeTab = 'customers'" class="px-4 py-2 rounded-xl font-bold text-sm transition-all" :class="activeTab === 'customers' ? 'bg-primary-600 text-white' : 'text-gray-500 hover:bg-gray-100'">
                                <i class="fa-solid fa-users ml-2"></i>مشتریان
                            </button>
                        </div>
                    </div>
                    
                    <!-- Orders Table -->
                    <div x-show="activeTab === 'orders'" class="overflow-x-auto">
                        <table class="w-full text-right border-collapse min-w-[800px]">
                            <thead class="bg-gray-50/80 text-gray-500 text-xs uppercase font-bold">
                                <tr>
                                    <th class="px-6 py-4">کد / تاریخ</th>
                                    <th class="px-6 py-4">مشتری</th>
                                    <th class="px-6 py-4">مبلغ</th>
                                    <th class="px-6 py-4">وضعیت</th>
                                    <th class="px-6 py-4 text-center">عملیات</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-100">
                                <template x-for="order in filteredOrders" :key="order.id">
                                    <tr class="hover:bg-gray-50/80 transition-colors">
                                        <td class="px-6 py-4">
                                            <div class="flex flex-col">
                                                <span class="font-mono font-bold text-gray-700" x-text="order.tracking_code"></span>
                                                <span class="text-xs text-gray-400 mt-1" x-text="order.created_at_persian"></span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="flex items-center gap-3">
                                                <div class="w-9 h-9 rounded-full bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center text-sm font-bold text-gray-600" x-text="order.customer_name ? order.customer_name.charAt(0) : '?'"></div>
                                                <div>
                                                    <div class="text-sm font-bold text-gray-800" x-text="order.customer_name"></div>
                                                    <div class="text-xs text-gray-500 font-mono" x-text="order.customer_phone"></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="font-black text-gray-800" x-text="formatPrice(order.total_price)"></span>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-xs font-bold border" :class="getStatus(order.status).class">
                                                <i class="fa-solid text-[10px]" :class="getStatus(order.status).icon"></i>
                                                <span x-text="getStatus(order.status).label"></span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <div class="flex justify-center items-center gap-2">
                                                <button @click="openInvoice(order.id)" class="w-8 h-8 rounded-lg text-gray-400 hover:text-emerald-600 hover:bg-emerald-50 flex items-center justify-center transition-all">
                                                    <i class="fa-solid fa-file-invoice"></i>
                                                </button>
                                                <a :href="'order_create.php?id=' + order.id" class="w-8 h-8 rounded-lg text-gray-400 hover:text-blue-600 hover:bg-blue-50 flex items-center justify-center transition-all">
                                                    <i class="fa-solid fa-pen-to-square"></i>
                                                </a>
                                                <button @click="sendWhatsApp(order)" class="w-8 h-8 rounded-lg text-gray-400 hover:text-green-600 hover:bg-green-50 flex items-center justify-center transition-all">
                                                    <i class="fa-brands fa-whatsapp"></i>
                                                </button>
                                                <div class="relative" x-data="{ open: false }">
                                                    <button @click="open = !open" @click.outside="open = false" class="w-8 h-8 rounded-lg text-gray-400 hover:text-gray-800 hover:bg-gray-100 flex items-center justify-center transition-all">
                                                        <i class="fa-solid fa-ellipsis-vertical"></i>
                                                    </button>
                                                    <div x-show="open" x-cloak x-transition class="absolute left-0 mt-2 w-48 bg-white rounded-xl border border-gray-100 shadow-xl z-50 py-1.5 text-right overflow-hidden">
                                                        <button @click="updateStatus(order.id, 'pending'); open = false" class="w-full text-right px-4 py-2 text-sm text-gray-600 hover:bg-amber-50 hover:text-amber-700 flex items-center gap-2"><div class="w-2 h-2 rounded-full bg-amber-500"></div> در انتظار</button>
                                                        <button @click="updateStatus(order.id, 'processing'); open = false" class="w-full text-right px-4 py-2 text-sm text-gray-600 hover:bg-blue-50 hover:text-blue-700 flex items-center gap-2"><div class="w-2 h-2 rounded-full bg-blue-500"></div> شستشو</button>
                                                        <button @click="updateStatus(order.id, 'ready'); open = false" class="w-full text-right px-4 py-2 text-sm text-gray-600 hover:bg-emerald-50 hover:text-emerald-700 flex items-center gap-2"><div class="w-2 h-2 rounded-full bg-emerald-500"></div> آماده</button>
                                                        <button @click="updateStatus(order.id, 'delivered'); open = false" class="w-full text-right px-4 py-2 text-sm text-gray-600 hover:bg-gray-100 flex items-center gap-2"><div class="w-2 h-2 rounded-full bg-gray-500"></div> تحویل شد</button>
                                                        <div class="border-t border-gray-100 my-1"></div>
                                                        <button @click="deleteOrder(order.id)" class="w-full text-right px-4 py-2 text-sm text-rose-600 hover:bg-rose-50"><i class="fa-solid fa-trash text-xs ml-2"></i> حذف</button>
                                                    </div>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                </template>
                            </tbody>
                        </table>
                    </div>

                    <!-- Customers Table -->
                    <div x-show="activeTab === 'customers'" class="overflow-x-auto">
                        <table class="w-full text-right border-collapse min-w-[800px]">
                            <thead class="bg-gray-50/80 text-gray-500 text-xs uppercase font-bold">
                                <tr>
                                    <th class="px-6 py-4">نام مشتری</th>
                                    <th class="px-6 py-4">شماره تماس</th>
                                    <th class="px-6 py-4">تعداد سفارش</th>
                                    <th class="px-6 py-4">مجموع خرید</th>
                                    <th class="px-6 py-4">آدرس</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-100">
                                <template x-for="customer in filteredCustomers" :key="customer.phone">
                                    <tr class="hover:bg-gray-50/80 transition-colors">
                                        <td class="px-6 py-4">
                                            <div class="flex items-center gap-3">
                                                <div class="w-9 h-9 rounded-full bg-gradient-to-br from-purple-100 to-pink-100 flex items-center justify-center text-sm font-bold text-purple-600" x-text="customer.name ? customer.name.charAt(0) : '?'"></div>
                                                <span class="text-sm font-bold text-gray-800" x-text="customer.name"></span>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="text-sm font-mono text-gray-700" x-text="customer.phone"></span>
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="bg-blue-100 text-blue-700 px-3 py-1 rounded-full text-xs font-bold" x-text="customer.orders_count"></span>
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="font-black text-emerald-600" x-text="formatPrice(customer.total_spent)"></span>
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="text-xs text-gray-500" x-text="customer.address || '-'"></span>
                                        </td>
                                    </tr>
                                </template>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Invoice Modal -->
    <div x-show="invoiceOpen" x-cloak class="fixed inset-0 z-[70] flex items-center justify-center p-4" style="display: none;">
        <div class="absolute inset-0 bg-gray-900/60 backdrop-blur-sm" @click="invoiceOpen = false"></div>
        <div class="relative bg-white rounded-3xl shadow-2xl max-w-4xl w-full max-h-[90vh] overflow-hidden" @click.stop>
            <iframe :src="'invoice.php?id=' + activeInvoiceId" class="w-full h-[90vh] border-0"></iframe>
            <button @click="invoiceOpen = false" class="absolute top-4 left-4 w-10 h-10 bg-gray-900 text-white rounded-full hover:bg-gray-800 transition flex items-center justify-center shadow-lg">
                <i class="fa-solid fa-xmark text-xl"></i>
            </button>
        </div>
    </div>

    <script>
        function adminDashboard() {
            return {
                orders: <?php echo json_encode($orders); ?>,
                customers: <?php echo json_encode($customers); ?>,
                search: '',
                invoiceOpen: false,
                activeInvoiceId: null,
                sidebarOpen: false,
                toasts: [],
                activeTab: 'orders',
                
                get filteredOrders() {
                    if (!this.search) return this.orders;
                    const q = this.search.toLowerCase();
                    return this.orders.filter(o => 
                        (o.customer_name && o.customer_name.toLowerCase().includes(q)) || 
                        (o.customer_phone && o.customer_phone.includes(q)) || 
                        (o.tracking_code && o.tracking_code.includes(q))
                    );
                },

                get filteredCustomers() {
                    if (!this.search) return this.customers;
                    const q = this.search.toLowerCase();
                    return this.customers.filter(c => 
                        (c.name && c.name.toLowerCase().includes(q)) || 
                        (c.phone && c.phone.includes(q))
                    );
                },

                openInvoice(orderId) {
                    this.activeInvoiceId = orderId;
                    this.invoiceOpen = true;
                },

                async updateStatus(id, newStatus) {
                    const order = this.orders.find(o => o.id === id);
                    const oldStatus = order.status;
                    order.status = newStatus;
                    this.showToast('وضعیت سفارش بروز شد');
                    try {
                        const res = await fetch(`../../api/orders.php?id=${id}`, {
                            method: 'PATCH',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify({status: newStatus})
                        });
                        if (!res.ok) throw new Error();
                        
                        // Auto send WhatsApp message
                        const autoSendRes = await fetch('../../api/settings.php?key=auto_send_enabled');
                        const autoSend = await autoSendRes.json();
                        
                        if (autoSend.value == '1') {
                            const msgKey = 'msg_' + newStatus;
                            const settingRes = await fetch(`../../api/settings.php?key=${msgKey}`);
                            if (settingRes.ok) {
                                const setting = await settingRes.json();
                                if (setting.value) {
                                    const message = setting.value
                                        .replace('{name}', order.customer_name)
                                        .replace('{code}', order.tracking_code)
                                        .replace('{price}', this.formatPrice(order.total_price))
                                        .replace('{phone}', order.customer_phone);
                                    
                                    console.log('Tracking code:', order.tracking_code);
                                    console.log('Message:', message);
                                    
                                    let phone = order.customer_phone.replace(/\D/g, '');
                                    if (phone.startsWith('0')) phone = '98' + phone.substring(1);
                                    else if (!phone.startsWith('98')) phone = '98' + phone;
                                    
                                    // Detect mobile or desktop
                                    const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                                    const whatsappUrl = isMobile 
                                        ? `https://api.whatsapp.com/send?phone=${phone}&text=${encodeURIComponent(message)}`
                                        : `https://web.whatsapp.com/send?phone=${phone}&text=${encodeURIComponent(message)}`;
                                    
                                    window.open(whatsappUrl, '_blank');
                                    this.showToast('پیام واتساپ آماده ارسال است');
                                }
                            }
                        }
                    } catch (e) {
                        order.status = oldStatus;
                        this.showToast('خطا در تغییر وضعیت', 'error');
                    }
                },

                async deleteOrder(id) {
                    if(!confirm('آیا از حذف این سفارش اطمینان دارید؟')) return;
                    try {
                        const res = await fetch(`../../api/orders.php?id=${id}`, { method: 'DELETE' });
                        if (res.ok) {
                            this.orders = this.orders.filter(o => o.id !== id);
                            this.showToast('سفارش حذف شد');
                        }
                    } catch (e) {
                        this.showToast('خطا در حذف سفارش', 'error');
                    }
                },

                sendWhatsApp(order) {
                    const phone = order.customer_phone.replace(/\D/g, '');
                    const invoiceUrl = `${window.location.origin}${window.location.pathname.replace('admin.php', 'invoice.php')}?id=${order.id}`;
                    const message = `سلام ${order.customer_name} عزیز\n\nفاکتور سفارش شما:\nکد پیگیری: ${order.tracking_code}\nمبلغ: ${this.formatPrice(order.total_price)} تومان\n\nمشاهده فاکتور:\n${invoiceUrl}`;
                    window.open(`https://api.whatsapp.com/send?phone=${phone}&text=${encodeURIComponent(message)}`, '_blank');
                },

                formatPrice(price) { return new Intl.NumberFormat('fa-IR').format(price); },

                getStatus(status) {
                    const configs = {
                        'pending': { label: 'در انتظار', class: 'bg-amber-50 text-amber-700 border-amber-200', icon: 'fa-clock' },
                        'processing': { label: 'شستشو', class: 'bg-blue-50 text-blue-700 border-blue-200', icon: 'fa-spinner fa-spin' },
                        'ready': { label: 'آماده', class: 'bg-emerald-50 text-emerald-700 border-emerald-200', icon: 'fa-check' },
                        'delivered': { label: 'تحویل شد', class: 'bg-gray-100 text-gray-600 border-gray-200', icon: 'fa-box' },
                        'cancelled': { label: 'لغو', class: 'bg-rose-50 text-rose-700 border-rose-200', icon: 'fa-xmark' }
                    };
                    return configs[status] || configs['pending'];
                },

                showToast(message, type = 'success') {
                    const id = Date.now();
                    this.toasts.push({ id, message, type, visible: true });
                    setTimeout(() => {
                        const t = this.toasts.find(t => t.id === id);
                        if(t) t.visible = false;
                    }, 3000);
                },
                
                async refreshOrders() {
                    try {
                        const res = await fetch('../../api/orders_list.php?t=' + Date.now());
                        const data = await res.json();
                        if (data.orders) {
                            this.orders = data.orders;
                        }
                    } catch (e) {
                        console.error('Failed to refresh orders:', e);
                    }
                }
            }
        }
        
        // Auto-refresh stats and orders every 5 seconds via AJAX
        setInterval(async () => {
            const dashboard = document.querySelector('[x-data]').__x.$data;
            
            try {
                // Update stats
                const statsRes = await fetch('../../api/stats.php?t=' + Date.now());
                const stats = await statsRes.json();
                
                document.querySelector('.bg-white.p-6.rounded-3xl:nth-child(1) .text-3xl').textContent = new Intl.NumberFormat('fa-IR').format(stats.today);
                document.querySelector('.bg-white.p-6.rounded-3xl:nth-child(2) .text-3xl').textContent = new Intl.NumberFormat('fa-IR').format(stats.pending);
                document.querySelector('.bg-white.p-6.rounded-3xl:nth-child(3) .text-3xl').textContent = new Intl.NumberFormat('fa-IR').format(stats.income);
                document.querySelector('.bg-white.p-6.rounded-3xl:nth-child(4) .text-3xl').textContent = new Intl.NumberFormat('fa-IR').format(stats.total);
                
                // Update orders list
                await dashboard.refreshOrders();
            } catch (e) {
                console.error('Failed to update:', e);
            }
        }, 5000);
    </script>
</body>
</html>
