<?php
session_start();
require_once '../../api/db.php';
require_once '../../api/persian_date.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

$order_id = $_GET['id'] ?? null;
if (!$order_id) {
    header("Location: admin.php");
    exit();
}

try {
    $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        header("Location: admin.php");
        exit();
    }

    // زمان ثبت
    $ts = $order['created_at'] > 10000000000 ? $order['created_at'] / 1000 : $order['created_at'];
    $order['created_at_persian'] = getPersianDate('Y/m/d', $ts);

    // تاریخ تحویل
    if (!empty($order['delivery_date'])) {
        if (is_numeric($order['delivery_date'])) {
            $delivery_ts = $order['delivery_date'] > 10000000000 ? $order['delivery_date'] / 1000 : $order['delivery_date'];
            $order['delivery_date_persian'] = getPersianDate('Y/m/d', $delivery_ts);
        } else {
            $order['delivery_date_persian'] = $order['delivery_date'];
        }
    } else {
        $order['delivery_date_persian'] = 'تعیین نشده';
    }

    $items = json_decode($order['items_json'], true) ?: [];

} catch (PDOException $e) {
    die("خطا در بارگذاری سفارش");
}

$status_map = [
    'pending'    => 'در انتظار تایید',
    'processing' => 'در حال شستشو',
    'ready'      => 'آماده تحویل',
    'delivered'  => 'تحویل شده',
    'cancelled'  => 'لغو شده'
];
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>فاکتور سفارش <?php echo htmlspecialchars($order['tracking_code']); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- فونت -->
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;700&display=swap" rel="stylesheet">
    <style>
        * {
            box-sizing: border-box;
        }
        :root {
            --primary: #4c51bf;
            --primary-light: #eef2ff;
            --border: #e5e7eb;
            --text: #111827;
            --muted: #6b7280;
        }
        body {
            margin: 0;
            padding: 70px 0 20px;
            font-family: "Vazirmatn", system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: #f3f4f6;
            color: var(--text);
        }

        /* نوار بالا (فقط در وب) */
        .toolbar {
            position: fixed;
            top: 0;
            right: 0;
            left: 0;
            background: #ffffff;
            border-bottom: 1px solid #e5e7eb;
            padding: 8px 16px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            z-index: 20;
        }
        .toolbar a,
        .toolbar button {
            font-size: 13px;
            border-radius: 999px;
            padding: 6px 12px;
            border: 1px solid #d1d5db;
            background: #ffffff;
            color: #374151;
            cursor: pointer;
        }
        .toolbar button {
            background: var(--text);
            color: #fff;
            border-color: var(--text);
        }

        /* کانتینر فاکتور */
        .invoice-wrapper {
            max-width: 900px;
            margin: 0 auto;
            padding: 8px;
        }
        .invoice {
            background: #ffffff;
            border-radius: 14px;
            box-shadow: 0 10px 30px rgba(15, 23, 42, 0.12);
            overflow: hidden;
            border: 1px solid #e5e7eb;
        }

        /* هدر */
        .invoice-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: 12px;
            padding: 14px 18px;
            background: linear-gradient(135deg, #4c51bf, #7c3aed);
            color: #fff;
        }
        .brand {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .brand-logo {
            width: 42px;
            height: 42px;
            border-radius: 12px;
            background: rgba(255,255,255,0.12);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            font-weight: 700;
        }
        .brand-text h1 {
            margin: 0;
            font-size: 18px;
            font-weight: 800;
        }
        .brand-text span {
            font-size: 11px;
            opacity: 0.85;
        }
        .header-meta {
            text-align: left;
            font-size: 11px;
        }
        .header-meta .label {
            font-size: 10px;
            opacity: 0.75;
        }
        .header-meta .value {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 999px;
            background: rgba(255,255,255,0.15);
            font-family: monospace;
            font-size: 11px;
        }

        /* بدنه */
        .invoice-body {
            padding: 14px 18px 16px;
        }

        .section-title {
            font-size: 11px;
            font-weight: 700;
            color: var(--primary);
            margin-bottom: 6px;
        }
        .card {
            border: 1px solid var(--border);
            border-radius: 10px;
            padding: 10px 12px;
            background: #f9fafb;
        }

        /* اطلاعات مشتری */
        .info-grid {
            display: grid;
            grid-template-columns: repeat(3, minmax(0, 1fr));
            gap: 8px 16px;
            font-size: 11px;
        }
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }
        .info-label {
            font-size: 10px;
            color: var(--muted);
        }
        .info-value {
            font-size: 11px;
            font-weight: 600;
        }
        .info-address {
            grid-column: 1 / -1;
            margin-top: 4px;
        }
        .info-address-text {
            font-size: 11px;
            white-space: pre-line;
        }

        /* جدول اقلام */
        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 11px;
        }
        thead {
            background: var(--primary-light);
        }
        th, td {
            padding: 6px 4px;
            border-bottom: 1px solid #e5e7eb;
        }
        th {
            text-align: right;
            font-weight: 700;
            color: #4b5563;
        }
        td {
            color: #374151;
        }
        tbody tr:nth-child(even) {
            background: #f9fafb;
        }
        .text-center {
            text-align: center;
        }
        .text-left {
            text-align: left;
        }
        .badge-qty {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 999px;
            background: #ecfdf5;
            color: #166534;
            font-weight: 700;
            font-size: 10px;
        }
        .total-row {
            margin-top: 6px;
            padding-top: 6px;
            border-top: 1px dashed #d1d5db;
            display: flex;
            justify-content: flex-end;
            align-items: baseline;
            gap: 8px;
            font-size: 12px;
        }
        .total-amount {
            font-weight: 800;
            font-size: 16px;
        }

        /* توضیحات */
        .note-box {
            border-radius: 8px;
            border: 1px dashed #f59e0b;
            background: #fffbeb;
            padding: 8px 10px;
            font-size: 11px;
            color: #92400e;
            white-space: pre-line;
        }

        /* فوتر */
        .invoice-footer {
            border-top: 1px dashed #e5e7eb;
            padding: 8px 12px 10px;
            text-align: center;
            font-size: 10px;
            color: var(--muted);
        }
        .invoice-footer strong {
            display: block;
            margin-bottom: 2px;
            font-size: 11px;
        }

        /* ریسپانسیو */
        @media (max-width: 640px) {
            body {
                padding-top: 60px;
                background: #ffffff;
            }
            .invoice-wrapper {
                padding: 4px;
            }
            .invoice {
                border-radius: 0;
                box-shadow: none;
                border-left: 0;
                border-right: 0;
            }
            .invoice-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .info-grid {
                grid-template-columns: 1fr 1fr;
            }
        }

        /* چاپ */
        @page {
            size: A5 landscape;
            margin: 5mm;
        }
        @media print {
            body {
                padding: 0;
                margin: 0;
                background: #ffffff;
            }
            .toolbar {
                display: none !important;
            }
            .invoice-wrapper {
                margin: 0;
                padding: 0;
            }
            .invoice {
                box-shadow: none;
                border-radius: 0;
                border: none;
                width: 100%;
                /* کوچک‌تر کردن محتوا روی برگه */
                transform: scale(0.9);
                transform-origin: top right;
            }
            .invoice-header {
                padding: 8px 10px;
            }
            .invoice-body {
                padding: 8px 10px 8px;
            }
            th, td {
                padding: 3px 2px;
                font-size: 9px;
            }
            .info-grid {
                gap: 4px 8px;
            }
            .info-label {
                font-size: 8px;
            }
            .info-value,
            .note-box {
                font-size: 9px;
            }
            .total-amount {
                font-size: 13px;
            }
        }
    </style>
</head>
<body>

<!-- نوار بالا (دکمه‌ها) -->
<div class="toolbar no-print">
    <a href="admin.php">بازگشت به مدیریت</a>
    <div style="display: flex; gap: 8px;">
        <button onclick="shareImage()">اشتراک گذاری</button>
        <button onclick="downloadImage()">دانلود عکس</button>
        <button onclick="downloadPDF()">دانلود PDF</button>
        <button onclick="window.print()">چاپ فاکتور</button>
    </div>
</div>

<div class="invoice-wrapper">
    <div class="invoice">

        <!-- هدر -->
        <div class="invoice-header">
            <div class="brand">
                <div class="brand-logo">🧼</div>
                <div class="brand-text">
                    <h1>پاکشو</h1>
                    <span>خشکشویی آنلاین</span>
                </div>
            </div>
            <div class="header-meta">
                <div>
                    <span class="label">شماره پیگیری:</span>
                    <span class="value"><?php echo htmlspecialchars($order['tracking_code']); ?></span>
                </div>
                <div style="margin-top:4px;">
                    <span class="label">تاریخ چاپ:</span>
                    <span><?php echo getPersianDate('Y/m/d', time()); ?></span>
                </div>
            </div>
        </div>

        <!-- بدنه -->
        <div class="invoice-body">

            <!-- اطلاعات سفارش و مشتری -->
            <div class="section">
                <div class="section-title">اطلاعات سفارش و مشتری</div>
                <div class="card">
                    <div class="info-grid">
                        <div class="info-item">
                            <span class="info-label">نام مشتری</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['customer_name']); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">شماره تماس</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['customer_phone']); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">تاریخ تحویل</span>
                            <span class="info-value"><?php echo $order['delivery_date_persian']; ?></span>
                        </div>

                        <?php if (!empty($order['customer_address'])): ?>
                        <div class="info-item">
                            <span class="info-label">آدرس</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['customer_address']); ?></span>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- اقلام سفارش -->
            <div class="section" style="margin-top: 12px;">
                <div class="section-title">اقلام سفارش</div>
                <div class="card">
                    <table>
                        <thead>
                            <tr>
                                <th style="width:40px;">ردیف</th>
                                <th>نام خدمت</th>
                                <th class="text-center" style="width:60px;">تعداد</th>
                                <th class="text-left" style="width:110px;">قیمت واحد</th>
                                <th class="text-left" style="width:120px;">جمع</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($items as $index => $item): ?>
                            <tr>
                                <td><?php echo $index + 1; ?></td>
                                <td><?php echo htmlspecialchars($item['name']); ?></td>
                                <td class="text-center">
                                    <span class="badge-qty"><?php echo (int)$item['qty']; ?></span>
                                </td>
                                <td class="text-left">
                                    <?php echo number_format($item['price']); ?> تومان
                                </td>
                                <td class="text-left">
                                    <?php echo number_format($item['price'] * $item['qty']); ?> تومان
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>

                    <div class="total-row">
                        <span>جمع کل قابل پرداخت:</span>
                        <span class="total-amount">
                            <?php echo number_format($order['total_price']); ?>
                        </span>
                        <span>تومان</span>
                    </div>
                </div>
            </div>

            <!-- توضیحات -->
            <?php if (!empty($order['notes'])): ?>
            <div class="section" style="margin-top: 10px;">
                <div class="section-title">توضیحات</div>
                <div class="note-box">
                    <?php echo nl2br(htmlspecialchars($order['notes'])); ?>
                </div>
            </div>
            <?php endif; ?>

        </div>

        <!-- فوتر -->
        <div class="invoice-footer">
            <strong>از انتخاب شما سپاسگزاریم</strong>
            خشکشویی آنلاین پاکشو
        </div>

    </div>
</div>

</body>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
<script>
function shareImage() {
    const element = document.querySelector('.invoice');
    html2canvas(element, { scale: 2, useCORS: true }).then(canvas => {
        canvas.toBlob(blob => {
            const file = new File([blob], 'invoice-<?php echo $order["tracking_code"]; ?>.png', { type: 'image/png' });
            if (navigator.share && navigator.canShare({ files: [file] })) {
                navigator.share({
                    files: [file],
                    title: 'فاکتور سفارش <?php echo $order["tracking_code"]; ?>',
                    text: 'فاکتور سفارش خشکشویی'
                }).catch(err => console.log(err));
            } else {
                alert('مرورگر شما از اشتراک گذاری پشتیبانی نمیکند');
            }
        });
    });
}

function downloadImage() {
    const element = document.querySelector('.invoice');
    html2canvas(element, { scale: 2, useCORS: true }).then(canvas => {
        const link = document.createElement('a');
        link.download = 'invoice-<?php echo $order["tracking_code"]; ?>.png';
        link.href = canvas.toDataURL('image/png');
        link.click();
    });
}

function downloadPDF() {
    const element = document.querySelector('.invoice');
    const opt = {
        margin: 5,
        filename: 'invoice-<?php echo $order["tracking_code"]; ?>.pdf',
        image: { type: 'jpeg', quality: 0.98 },
        html2canvas: { scale: 2, useCORS: true },
        jsPDF: { unit: 'mm', format: 'a5', orientation: 'landscape' }
    };
    html2pdf().set(opt).from(element).save();
}
</script>
</html>
