<?php
session_start();
header("Cache-Control: no-cache, no-store, must-revalidate");
require_once '../../api/db.php';

// امنیت و بررسی دسترسی
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

// بررسی حالت ویرایش
$editMode = isset($_GET['id']);
$orderData = null;
if ($editMode) {
    try {
        $stmt = $conn->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $orderData = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$orderData) {
            header("Location: admin.php");
            exit();
        }
    } catch(PDOException $e) {
        header("Location: admin.php");
        exit();
    }
}

// دریافت خدمات
try {
    $stmt = $conn->query("SELECT * FROM services ORDER BY id DESC");
    $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    $services = [];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl" class="scroll-smooth">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $editMode ? 'ویرایش سفارش' : 'ثبت سفارش جدید'; ?> | پنل مدیریت</title>

    <script src="https://cdn.tailwindcss.com"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">

    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        sans: ['Vazirmatn', 'system-ui', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50:  '#f5f3ff',
                            100: '#ede9fe',
                            200: '#ddd6fe',
                            400: '#a855f7',
                            500: '#8b5cf6',
                            600: '#7c3aed',
                        },
                        mint: {
                            100: '#e0fbf4',
                            400: '#2dd4bf',
                        },
                    },
                    boxShadow: {
                        glass: '0 18px 45px rgba(15,23,42,0.15)',
                        card: '0 14px 35px rgba(148,163,184,0.25)',
                    },
                    borderRadius: {
                        '2xl': '1.25rem',
                        '3xl': '1.75rem',
                    },
                }
            }
        }
    </script>

    <style>
        [x-cloak] { display:none !important; }
        ::-webkit-scrollbar { width: 6px; height: 6px; }
        ::-webkit-scrollbar-track { background: transparent; }
        ::-webkit-scrollbar-thumb { background: #cbd5e1; border-radius: 999px; }
    </style>
</head>

<body class="font-sans bg-gradient-to-br from-sky-50 via-slate-50 to-primary-50 text-slate-800 antialiased"
      x-data="orderSystem()" x-init="initSystem()">

<!-- Toast -->
<div class="fixed top-4 left-1/2 -translate-x-1/2 z-[9999]" x-cloak>
    <div x-show="toast.visible"
         x-transition:enter="transition ease-out duration-200"
         x-transition:enter-start="opacity-0 -translate-y-3"
         x-transition:enter-end="opacity-100 translate-y-0"
         x-transition:leave="transition ease-in duration-150"
         x-transition:leave-end="opacity-0 -translate-y-3"
         :class="toast.type === 'success' ? 'bg-emerald-500 text-white' : 'bg-rose-500 text-white'"
         class="px-5 py-3 rounded-2xl shadow-card flex items-center gap-3 min-w-[260px] justify-center backdrop-blur-md border border-white/40">
        <i :class="toast.type === 'success' ? 'fa-solid fa-circle-check' : 'fa-solid fa-triangle-exclamation'" class="text-lg"></i>
        <span x-text="toast.message" class="font-semibold text-sm"></span>
    </div>
</div>

<div class="min-h-screen px-3 sm:px-4 lg:px-8 py-4 sm:py-6 flex justify-center">
    <div class="w-full max-w-7xl bg-white/80 shadow-glass border border-white/70 rounded-3xl overflow-hidden backdrop-blur-md flex flex-col lg:flex-row">

        <!-- Left Side: Content -->
        <div class="flex-1 flex flex-col">
            <!-- Topbar (Desktop) -->
            <header class="hidden sm:flex items-center justify-between px-4 sm:px-6 pt-4 pb-3 border-b border-slate-100 bg-white/80 backdrop-blur">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-2xl bg-gradient-to-tr from-primary-500 to-mint-400 text-white flex items-center justify-center shadow-card">
                        <i class="fa-solid fa-gem text-lg"></i>
                    </div>
                    <div>
                        <div class="inline-flex items-center gap-1.5 text-[11px] px-2.5 py-1 rounded-full bg-primary-50 text-primary-600 border border-primary-100">
                            <span class="w-1.5 h-1.5 rounded-full bg-emerald-400 animate-pulse"></span>
                            <span><?php echo $editMode ? 'حالت ویرایش' : 'سفارش جدید فعال'; ?></span>
                        </div>
                        <h1 class="mt-1 text-xl font-extrabold"><?php echo $editMode ? 'ویرایش سفارش' : 'ثبت سفارش جدید'; ?></h1>
                    </div>
                </div>
                <div class="flex items-center gap-2">
                    <a href="services.php" class="hidden md:inline-flex items-center gap-1.5 text-xs px-3 py-1.5 rounded-full border border-slate-200 hover:border-primary-400 hover:text-primary-600 transition-colors">
                        <i class="fa-solid fa-layer-group text-[11px]"></i>
                        مدیریت خدمات
                    </a>
                    <a href="admin.php" class="w-9 h-9 rounded-2xl bg-slate-100 text-slate-600 flex items-center justify-center hover:bg-slate-200 transition-colors">
                        <i class="fa-solid fa-list-check text-sm"></i>
                    </a>
                    <a href="../../logout.php" class="w-9 h-9 rounded-2xl bg-rose-50 text-rose-500 flex items-center justify-center hover:bg-rose-100 transition-colors">
                        <i class="fa-solid fa-power-off text-sm"></i>
                    </a>
                </div>
            </header>

            <!-- Topbar (Mobile) -->
            <header class="sm:hidden flex items-center justify-between px-3 pt-3 pb-2 bg-white/90 border-b border-slate-100">
                <div class="flex items-center gap-2">
                    <div class="w-9 h-9 rounded-2xl bg-gradient-to-tr from-primary-500 to-mint-400 text-white flex items-center justify-center shadow-card">
                        <i class="fa-solid fa-gem text-sm"></i>
                    </div>
                    <div>
                        <h1 class="text-sm font-extrabold"><?php echo $editMode ? 'ویرایش سفارش' : 'ثبت سفارش'; ?></h1>
                    </div>
                </div>
                <div class="flex items-center gap-1.5">
                    <button class="relative" @click="showMobileCart = true">
                        <div class="w-9 h-9 rounded-2xl bg-slate-100 flex items-center justify-center text-slate-600">
                            <i class="fa-solid fa-basket-shopping text-sm"></i>
                        </div>
                        <span x-show="cartTotalQty > 0" x-text="cartTotalQty" class="absolute -top-1 -right-1 w-4.5 h-4.5 bg-primary-500 text-white text-[10px] rounded-full flex items-center justify-center font-bold ring-2 ring-white"></span>
                    </button>
                    <a href="admin.php" class="w-9 h-9 rounded-2xl bg-slate-100 text-slate-600 flex items-center justify-center">
                        <i class="fa-solid fa-arrow-left text-sm"></i>
                    </a>
                </div>
            </header>

            <main class="flex-1 overflow-y-auto px-3 sm:px-5 lg:px-6 pb-5 space-y-5 sm:space-y-6">
                <!-- Search -->
                <section class="pt-3 sm:pt-4 space-y-3">
                    <div class="flex flex-col lg:flex-row lg:items-center gap-3">
                        <div class="flex-1 relative group">
                            <i class="fa-solid fa-magnifying-glass absolute right-3 top-1/2 -translate-y-1/2 text-slate-400 group-focus-within:text-primary-500 transition-colors"></i>
                            <input type="text" x-model="searchQuery" placeholder="جستجو در بین خدمات..." class="w-full h-11 rounded-2xl bg-slate-50 border border-slate-200 text-sm pr-9 pl-3 placeholder:text-slate-400 focus:bg-white focus:border-primary-500 focus:ring-2 focus:ring-primary-500/15 outline-none">
                        </div>
                    </div>
                </section>

                <!-- Customer Form -->
                <section class="bg-white/90 border border-slate-100 rounded-3xl shadow-card px-4 sm:px-5 pt-4 pb-5">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center gap-2.5">
                            <div class="w-9 h-9 rounded-2xl bg-primary-50 text-primary-600 flex items-center justify-center">
                                <i class="fa-solid fa-user-pen text-base"></i>
                            </div>
                            <p class="text-sm font-semibold">مشخصات مشتری</p>
                        </div>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="space-y-1.5">
                            <label class="block text-[11px] font-medium text-slate-600">شماره موبایل <span class="text-rose-500">*</span></label>
                            <input type="tel" x-model="customer.phone" @input="fetchUserByPhone()" class="w-full h-11 px-3 rounded-2xl bg-slate-50 border border-slate-200 focus:bg-white focus:border-primary-500 outline-none text-left dir-ltr text-sm" placeholder="09xx ...">
                        </div>
                        <div class="space-y-1.5">
                            <label class="block text-[11px] font-medium text-slate-600">نام و نام خانوادگی <span class="text-rose-500">*</span></label>
                            <input type="text" x-model="customer.name" class="w-full h-11 px-3 rounded-2xl bg-slate-50 border border-slate-200 focus:bg-white focus:border-primary-500 outline-none text-sm" placeholder="مثال: علی محمدی">
                        </div>
                        <div class="md:col-span-2 space-y-1.5">
                            <label class="block text-[11px] font-medium text-slate-600">آدرس (اختیاری)</label>
                            <textarea rows="2" x-model="customer.address" class="w-full p-3 rounded-2xl bg-slate-50 border border-slate-200 focus:bg-white focus:border-primary-500 outline-none text-sm resize-none"></textarea>
                        </div>
                        <div class="space-y-1.5">
                            <label class="block text-[11px] font-medium text-slate-600">تاریخ تحویل (اختیاری)</label>
                            <div class="relative">
                                <input type="text" x-model="deliveryDate" @click="showDatePicker = true" readonly placeholder="انتخاب تاریخ تحویل" class="w-full h-11 px-3 rounded-2xl bg-slate-50 border border-slate-200 focus:bg-white focus:border-primary-500 outline-none text-sm text-center cursor-pointer">
                                <i class="fa-solid fa-calendar-days absolute left-3 top-1/2 -translate-y-1/2 text-slate-400 pointer-events-none"></i>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- Services Grid -->
                <section class="space-y-3 pb-3">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center gap-2">
                            <div class="w-8 h-8 rounded-2xl bg-mint-100 text-mint-400 flex items-center justify-center">
                                <i class="fa-solid fa-grid-2 text-sm"></i>
                            </div>
                            <p class="text-sm font-semibold">انتخاب خدمات</p>
                        </div>
                    </div>

                    <div class="grid grid-cols-2 md:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
                        <template x-for="service in filteredServices" :key="service.id">
                            <button type="button" @click="addToCart(service)" class="group relative overflow-hidden rounded-3xl bg-white/90 border border-slate-100 hover:border-primary-300 hover:-translate-y-0.5 hover:shadow-card transition-all p-3.5 sm:p-4 text-right text-slate-700">
                                <div x-show="isInCart(service.id)" class="absolute top-2.5 left-2.5 flex items-center gap-1 rounded-full bg-primary-500 text-white text-[10px] px-2 py-1 shadow-md">
                                    <i class="fa-solid fa-circle-check text-[9px]"></i>
                                    <span x-text="'تعداد: ' + getQty(service.id)"></span>
                                </div>
                                <div class="relative flex flex-col items-center text-center gap-2">
                                    <div class="w-12 h-12 sm:w-14 sm:h-14 rounded-2xl bg-slate-50 border border-slate-100 flex items-center justify-center text-primary-500 text-xl group-hover:scale-105 transition-all">
                                        <template x-if="service.icon_url">
                                            <img :src="'../../' + service.icon_url" class="w-10 h-10 object-contain rounded-xl">
                                        </template>
                                        <template x-if="!service.icon_url">
                                            <i :class="'fa-solid ' + (service.icon || 'fa-box')"></i>
                                        </template>
                                    </div>
                                    <div class="w-full space-y-1">
                                        <h3 class="font-semibold text-[13px] sm:text-sm text-slate-800 line-clamp-1" x-text="service.name"></h3>
                                        <div class="flex items-center justify-center gap-1 text-primary-600 font-extrabold text-[13px] sm:text-sm">
                                            <span x-text="formatNumber(service.price)"></span>
                                            <span class="text-[10px] font-normal text-slate-400">تومان</span>
                                        </div>
                                    </div>
                                </div>
                            </button>
                        </template>
                    </div>

                    <div x-show="filteredServices.length === 0" class="flex flex-col items-center justify-center py-10 text-center">
                        <p class="text-sm font-medium text-slate-600">خدمتی پیدا نشد</p>
                    </div>
                </section>
            </main>
        </div>

        <!-- Right Side: Invoice (Desktop) -->
        <aside class="hidden md:flex w-[320px] xl:w-[360px] flex-col bg-gradient-to-b from-white via-slate-50 to-sky-50 border-r border-slate-100 relative">
            <div class="relative p-5 border-b border-slate-100 bg-white/80 backdrop-blur-sm">
                <div class="flex items-center justify-between mb-3">
                    <div>
                        <p class="text-xs text-slate-500">فاکتور سفارش</p>
                        <h2 class="font-extrabold text-lg tracking-tight text-slate-800">پیش‌فاکتور</h2>
                    </div>
                    <span class="inline-flex items-center gap-1 text-[11px] px-2.5 py-1 rounded-full bg-slate-50 border border-slate-200 font-mono text-slate-600" x-text="trackingCode"></span>
                </div>
            </div>

            <div class="relative flex-1 overflow-y-auto p-4 space-y-3">
                <template x-for="item in cart" :key="item.id">
                    <div class="group flex items-center gap-3 rounded-2xl bg-white/90 border border-slate-100 p-3 shadow-sm hover:shadow-card transition-all">
                        <div class="flex-1 min-w-0">
                            <div class="flex items-center justify-between gap-2 mb-1">
                                <div class="font-semibold text-[13px] truncate text-slate-800" x-text="item.name"></div>
                            </div>
                            <div class="flex items-center justify-between text-[11px] text-slate-400">
                                <span class="text-primary-500" x-text="formatNumber(item.qty * item.price) + ' تومان'"></span>
                            </div>
                        </div>
                        <div class="flex items-center gap-1.5 bg-slate-50 rounded-2xl border border-slate-200 px-1.5 py-1">
                            <button @click.stop="increaseQty(item.id)" class="w-6 h-6 rounded-xl bg-primary-50 text-primary-600 hover:bg-primary-500 hover:text-white text-[10px] flex items-center justify-center transition-colors"><i class="fa-solid fa-plus"></i></button>
                            <span class="w-4 text-center text-xs font-semibold" x-text="item.qty"></span>
                            <button @click.stop="decreaseQty(item.id)" class="w-6 h-6 rounded-xl bg-slate-100 text-slate-500 hover:bg-rose-500 hover:text-white text-[10px] flex items-center justify-center transition-colors"><i class="fa-solid fa-minus"></i></button>
                        </div>
                    </div>
                </template>
                <div x-show="cart.length === 0" class="flex flex-col items-center justify-center h-60 text-slate-400">
                    <span class="text-sm font-medium">سبد خرید خالی است</span>
                </div>
            </div>

            <div class="relative p-4 border-t border-slate-100 bg-white/85 backdrop-blur-sm">
                <div class="mb-3">
                    <textarea rows="2" x-model="notes" class="w-full text-[12px] bg-slate-50 border border-slate-200 rounded-2xl px-3 py-2 outline-none focus:border-primary-500 resize-none" placeholder="یادداشت سفارش..."></textarea>
                </div>
                <div class="flex items-end justify-between mb-3">
                    <div class="space-y-1 text-[11px] text-slate-500">
                        <p>مبلغ قابل پرداخت</p>
                    </div>
                    <div class="text-right">
                        <div class="text-2xl font-black tracking-tight text-slate-900">
                            <span x-text="formatNumber(totalPrice)"></span>
                        </div>
                        <div class="text-[11px] text-slate-500">تومان</div>
                    </div>
                </div>
                <button @click="submitOrder()" :disabled="cart.length === 0 || isLoading" class="w-full h-11 rounded-2xl bg-gradient-to-r from-primary-500 to-mint-400 text-white font-bold text-sm flex items-center justify-center gap-2 shadow-card disabled:opacity-50 transition-all">
                    <i class="fa-solid" :class="isLoading ? 'fa-circle-notch fa-spin' : 'fa-check'"></i>
                    <span x-text="isLoading ? 'در حال ثبت...' : (editMode ? 'بروزرسانی سفارش' : 'ثبت نهایی سفارش')"></span>
                </button>
            </div>
        </aside>

        <!-- Mobile Cart Sheet -->
        <div class="md:hidden fixed inset-0 z-40" x-show="showMobileCart" x-cloak>
            <div class="absolute inset-0 bg-black/30 backdrop-blur-sm" @click="showMobileCart = false"></div>
            <div class="absolute bottom-0 left-0 right-0 bg-white rounded-t-3xl shadow-2xl max-h-[85vh] flex flex-col border-t border-slate-100">
                <div class="w-10 h-1.5 bg-slate-300 rounded-full mx-auto mt-3 mb-2"></div>
                <div class="px-4 pb-3 pt-1 border-b border-slate-100 flex items-center justify-between">
                    <h2 class="font-semibold text-sm">سبد خرید</h2>
                    <span class="px-2.5 py-1 rounded-full text-[11px] bg-slate-50 text-slate-600" x-text="cartTotalQty + ' مورد'"></span>
                </div>
                <div class="flex-1 overflow-y-auto p-4 space-y-3">
                    <template x-for="item in cart" :key="item.id">
                        <div class="flex justify-between items-center p-3 border border-slate-100 rounded-2xl bg-slate-50">
                            <div>
                                <div class="font-semibold text-[13px]" x-text="item.name"></div>
                                <div class="text-[11px] text-slate-500 mt-0.5"><span x-text="formatNumber(item.price)"></span> تومان</div>
                            </div>
                            <div class="flex items-center gap-3">
                                <button @click="decreaseQty(item.id)" class="w-8 h-8 rounded-full bg-rose-50 text-rose-500 flex items-center justify-center"><i class="fa-solid fa-minus text-xs"></i></button>
                                <span class="font-semibold text-sm" x-text="item.qty"></span>
                                <button @click="increaseQty(item.id)" class="w-8 h-8 rounded-full bg-emerald-50 text-emerald-500 flex items-center justify-center"><i class="fa-solid fa-plus text-xs"></i></button>
                            </div>
                        </div>
                    </template>
                </div>
                <div class="p-4 bg-white border-t border-slate-100 space-y-2">
                    <button @click="showMobileCart = false; submitOrder()" :disabled="cart.length === 0 || isLoading" class="w-full h-11 bg-gradient-to-r from-primary-500 to-mint-400 text-white rounded-2xl font-bold text-sm shadow-card flex items-center justify-center gap-2">
                         <span x-text="isLoading ? 'در حال ثبت...' : ('پرداخت ' + formatNumber(totalPrice) + ' تومان')"></span>
                    </button>
                </div>
            </div>
        </div>

    </div>
</div>

<!-- Date Picker Modal (Corrected) -->
<div x-show="showDatePicker" @click.self="showDatePicker = false" class="fixed inset-0 z-[9999] flex items-center justify-center bg-black/30 backdrop-blur-sm" x-cloak>
    <div class="bg-white rounded-2xl shadow-2xl p-6 w-80" @click.stop>
        <div class="flex items-center justify-between mb-4 pb-3 border-b border-slate-200">
            <button type="button" @click="prevMonth()" class="w-9 h-9 rounded-xl hover:bg-slate-100 flex items-center justify-center transition-colors">
                <i class="fa-solid fa-chevron-right"></i>
            </button>
            <span class="font-bold text-lg" x-text="monthNames[currentMonth] + ' ' + currentYear"></span>
            <button type="button" @click="nextMonth()" class="w-9 h-9 rounded-xl hover:bg-slate-100 flex items-center justify-center transition-colors">
                <i class="fa-solid fa-chevron-left"></i>
            </button>
        </div>
        <div class="grid grid-cols-7 gap-1 mb-2">
            <template x-for="day in ['ش','ی','د','س','چ','پ','ج']">
                <div class="text-center text-xs font-bold text-slate-400 py-2" x-text="day"></div>
            </template>
        </div>
        <div class="grid grid-cols-7 gap-1">
            <template x-for="(day, index) in calendarDays" :key="index">
                <button type="button" @click="selectDate(day)" 
                     :class="{
                         'aspect-square flex items-center justify-center rounded-lg text-sm transition-all': true,
                         'bg-primary-500 text-white font-bold': day.selected,
                         'border-2 border-primary-500': day.isToday && !day.selected,
                         'opacity-30 cursor-not-allowed': day.disabled || !day.currentMonth,
                         'hover:bg-slate-100': !day.disabled && day.currentMonth && !day.selected,
                         'text-slate-300': !day.currentMonth,
                         'text-slate-700': day.currentMonth
                     }"
                     :disabled="day.disabled || !day.currentMonth"
                     x-text="day.day"></button>
            </template>
        </div>
    </div>
</div>

<script>
    function orderSystem() {
        return {
            services: <?php echo json_encode($services); ?>,
            editMode: <?php echo $editMode ? 'true' : 'false'; ?>,
            orderId: <?php echo $editMode ? $orderData['id'] : 'null'; ?>,
            cart: [],
            searchQuery: '',
            isLoading: false,
            showMobileCart: false,
            notes: <?php echo $editMode ? json_encode($orderData['notes'] ?? '') : "''" ?>,
            trackingCode: <?php echo $editMode ? json_encode($orderData['tracking_code']) : "''" ?>,
            customer: { 
                name: <?php echo $editMode ? json_encode($orderData['customer_name']) : "''" ?>, 
                phone: <?php echo $editMode ? json_encode($orderData['customer_phone']) : "''" ?>, 
                address: <?php echo $editMode ? json_encode($orderData['customer_address'] ?? '') : "''" ?> 
            },
            phoneTimeout: null,
            
            // Calendar State
            deliveryDate: '',
            showDatePicker: false,
            currentYear: 1403,
            currentMonth: 9,
            selectedDate: null,
            monthNames: ['فروردین','اردیبهشت','خرداد','تیر','مرداد','شهریور','مهر','آبان','آذر','دی','بهمن','اسفند'],
            calendarDays: [],
            
            toast: { visible: false, message: '', type: 'success' },

            initSystem() {
                if (!this.editMode) {
                    this.generateTrackingCode();
                } else {
                    const items = <?php echo $editMode ? $orderData['items_json'] : '[]'; ?>;
                    this.cart = items;
                    const deliveryDate = <?php echo $editMode && !empty($orderData['delivery_date']) ? json_encode($orderData['delivery_date']) : 'null'; ?>;
                    if (deliveryDate) {
                        this.deliveryDate = deliveryDate;
                        this.selectedDate = deliveryDate;
                    }
                }
                const today = new Date();
                const jToday = this.gregorianToJalali(today.getFullYear(), today.getMonth() + 1, today.getDate());
                this.currentYear = jToday[0];
                this.currentMonth = jToday[1] - 1;
                this.loadCalendar();
            },

            async fetchUserByPhone() {
                clearTimeout(this.phoneTimeout);
                if (this.customer.phone.length < 11) return;
                
                this.phoneTimeout = setTimeout(async () => {
                    try {
                        const res = await fetch(`../../api/users.php?phone=${this.customer.phone}`);
                        if (res.ok) {
                            const data = await res.json();
                            if (data.full_name) {
                                this.customer.name = data.full_name;
                                this.customer.address = data.address || '';
                            }
                        }
                    } catch (e) {}
                }, 500);
            },

            get filteredServices() {
                if (!this.searchQuery) return this.services;
                const q = this.searchQuery.toLowerCase();
                return this.services.filter(s => s.name.toLowerCase().includes(q));
            },

            get totalPrice() {
                return this.cart.reduce((sum, item) => sum + (item.price * item.qty), 0);
            },

            get cartTotalQty() {
                return this.cart.reduce((sum, item) => sum + item.qty, 0);
            },

            addToCart(service) {
                const existing = this.cart.find(i => i.id === service.id);
                if (existing) {
                    existing.qty++;
                } else {
                    this.cart.push({ ...service, qty: 1 });
                }
                if (navigator.vibrate) navigator.vibrate(40);
            },

            increaseQty(id) {
                const item = this.cart.find(i => i.id === id);
                if (item) item.qty++;
            },

            decreaseQty(id) {
                const item = this.cart.find(i => i.id === id);
                if (item) {
                    item.qty--;
                    if (item.qty <= 0) {
                        this.cart = this.cart.filter(i => i.id !== id);
                    }
                }
            },

            getQty(id) {
                const item = this.cart.find(i => i.id === id);
                return item ? item.qty : 0;
            },

            isInCart(id) {
                return this.cart.some(i => i.id === id);
            },

            formatNumber(num) {
                return new Intl.NumberFormat('fa-IR').format(num);
            },

            generateTrackingCode() {
                const now = new Date();
                const jDate = this.gregorianToJalali(now.getFullYear(), now.getMonth() + 1, now.getDate());
                const year = jDate[0].toString();
                const month = jDate[1].toString().padStart(2, '0');
                const random = Math.floor(1000 + Math.random() * 9000);
                this.trackingCode = `${year}${month}-${random}`;
            },

            showToast(msg, type = 'success') {
                this.toast.message = msg;
                this.toast.type = type;
                this.toast.visible = true;
                setTimeout(() => this.toast.visible = false, 2600);
            },

            async submitOrder() {
                if (!this.customer.name || !this.customer.phone) {
                    this.showToast('لطفا نام و شماره تماس را وارد کنید', 'error');
                    return;
                }
                if (this.cart.length === 0) {
                    this.showToast('حداقل یک خدمت را انتخاب کنید', 'error');
                    return;
                }

                this.isLoading = true;
                const payload = {
                    user_id: <?php echo $_SESSION['user_id']; ?>,
                    customer_name: this.customer.name,
                    customer_phone: this.customer.phone,
                    customer_address: this.customer.address,
                    tracking_code: this.trackingCode,
                    items_json: JSON.stringify(this.cart),
                    total_price: this.totalPrice,
                    notes: this.notes,
                    delivery_date: this.deliveryDate || null
                };

                try {
                    const url = this.editMode ? `../../api/orders.php?id=${this.orderId}` : '../../api/orders.php';
                    const method = this.editMode ? 'PUT' : 'POST';
                    const response = await fetch(url, {
                        method: method,
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify(payload)
                    });

                    if (response.ok) {
                        this.showToast(this.editMode ? 'سفارش بروزرسانی شد' : 'سفارش با موفقیت ثبت شد', 'success');
                        setTimeout(() => window.location.href = 'admin.php', 1300);
                    } else {
                        throw new Error('Server Error');
                    }
                } catch (e) {
                    this.showToast('خطا در ارتباط با سرور', 'error');
                    this.isLoading = false;
                }
            },

            // --- منطق اصلاح شده تقویم ---
            loadCalendar() {
                const days = [];
                let daysInMonth;
                if (this.currentMonth < 6) daysInMonth = 31;
                else if (this.currentMonth < 11) daysInMonth = 30;
                else daysInMonth = this.isLeapJalali(this.currentYear) ? 30 : 29;

                const jDate1 = this.jalaliToGregorian(this.currentYear, this.currentMonth + 1, 1);
                const gDate = new Date(jDate1[0], jDate1[1] - 1, jDate1[2]);
                let dayOfWeek = gDate.getDay(); 
                let startDayIndex = (dayOfWeek + 1) % 7; 

                const today = new Date();
                const jToday = this.gregorianToJalali(today.getFullYear(), today.getMonth() + 1, today.getDate());

                for (let i = 0; i < startDayIndex; i++) {
                    days.push({ day: '', currentMonth: false });
                }

                for (let i = 1; i <= daysInMonth; i++) {
                    const dateStr = `${this.currentYear}/${String(this.currentMonth + 1).padStart(2, '0')}/${String(i).padStart(2, '0')}`;
                    let isPast = false;
                    if (this.currentYear < jToday[0]) isPast = true;
                    else if (this.currentYear === jToday[0] && (this.currentMonth + 1) < jToday[1]) isPast = true;
                    else if (this.currentYear === jToday[0] && (this.currentMonth + 1) === jToday[1] && i < jToday[2]) isPast = true;

                    days.push({
                        day: i,
                        date: dateStr,
                        currentMonth: true,
                        selected: this.selectedDate === dateStr,
                        isToday: this.isToday(i),
                        disabled: isPast
                    });
                }
                this.calendarDays = days;
            },

            isToday(day) {
                const today = new Date();
                const jToday = this.gregorianToJalali(today.getFullYear(), today.getMonth() + 1, today.getDate());
                return jToday[0] === this.currentYear && jToday[1] === this.currentMonth + 1 && jToday[2] === day;
            },

            selectDate(day) {
                if (!day.currentMonth || day.disabled) return;
                this.selectedDate = day.date;
                this.deliveryDate = day.date;
                this.showDatePicker = false;
            },

            prevMonth() { 
                if (this.currentMonth === 0) {
                    this.currentMonth = 11;
                    this.currentYear--;
                } else {
                    this.currentMonth--;
                }
                this.loadCalendar();
            },

            nextMonth() { 
                if (this.currentMonth === 11) {
                    this.currentMonth = 0;
                    this.currentYear++;
                } else {
                    this.currentMonth++;
                }
                this.loadCalendar();
            },

            // توابع دقیق تبدیل تاریخ
            jalaliToGregorian(jy, jm, jd) {
                var g_y, g_m, g_d;
                var jy_calc = jy - 979;
                var j_day_no = 365 * jy_calc + parseInt(jy_calc / 33) * 8 + parseInt((jy_calc % 33 + 3) / 4);
                for (var i = 0; i < jm - 1; ++i) j_day_no += (i < 6) ? 31 : 30;
                j_day_no += jd - 1;
                var g_day_no = j_day_no + 79;
                var gy = 1600 + 400 * parseInt(g_day_no / 146097);
                g_day_no = g_day_no % 146097;
                var leap = true;
                if (g_day_no >= 36525) {
                    g_day_no--;
                    gy += 100 * parseInt(g_day_no / 36524);
                    g_day_no = g_day_no % 36524;
                    if (g_day_no >= 365) g_day_no++;
                    else leap = false;
                }
                gy += 4 * parseInt(g_day_no / 1461);
                g_day_no %= 1461;
                if (g_day_no >= 366) {
                    leap = false;
                    g_day_no--;
                    gy += parseInt(g_day_no / 365);
                    g_day_no = g_day_no % 365;
                }
                for (var i = 0; g_day_no >= ((i==1 && leap) ? 29 : [31,28,31,30,31,30,31,31,30,31,30,31][i]); i++)
                    g_day_no -= ((i==1 && leap) ? 29 : [31,28,31,30,31,30,31,31,30,31,30,31][i]);
                g_m = i + 1;
                g_d = g_day_no + 1;
                return [gy, g_m, g_d];
            },

            gregorianToJalali(g_y, g_m, g_d) {
                var g_days_in_month = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
                var j_days_in_month = [31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29];
                var gy = g_y - 1600;
                var gm = g_m - 1;
                var gd = g_d - 1;
                var g_day_no = 365 * gy + parseInt((gy + 3) / 4) - parseInt((gy + 99) / 100) + parseInt((gy + 399) / 400);
                for (var i = 0; i < gm; ++i) g_day_no += g_days_in_month[i];
                if (gm > 1 && ((gy % 4 == 0 && gy % 100 != 0) || (gy % 400 == 0))) g_day_no++;
                g_day_no += gd;
                var j_day_no = g_day_no - 79;
                var j_np = parseInt(j_day_no / 12053);
                j_day_no %= 12053;
                var jy = 979 + 33 * j_np + 4 * parseInt(j_day_no / 1461);
                j_day_no %= 1461;
                if (j_day_no >= 366) {
                    jy += parseInt((j_day_no - 1) / 365);
                    j_day_no = (j_day_no - 1) % 365;
                }
                for (var i = 0; i < 11 && j_day_no >= j_days_in_month[i]; ++i) j_day_no -= j_days_in_month[i];
                var jm = i + 1;
                var jd = j_day_no + 1;
                return [jy, jm, jd];
            },

            isLeapJalali(jy) {
                var r = ((jy - (jy > 0 ? 474 : 473)) % 2820 + 474 + 38) * 682;
                var t = (r % 2816);
                return (t < 682);
            }
        }
    }
</script>

</body>
</html>
