<?php
session_start();
require_once '../../api/db.php';
require_once '../../api/persian_date.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'user') {
    header("Location: ../../login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// دریافت سفارشات کاربر
try {
    $stmt = $conn->prepare("SELECT * FROM orders WHERE user_id = :uid ORDER BY created_at DESC");
    $stmt->execute([':uid' => $user_id]);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // تبدیل تاریخ به شمسی
    foreach($orders as &$o) {
        $ts = $o['created_at'] > 10000000000 ? $o['created_at'] / 1000 : $o['created_at'];
        $o['created_at_persian'] = getPersianDate('Y/m/d H:i', $ts);
    }
    
    // محاسبه آمار
    $stats = [
        'total'      => count($orders),
        'total_spent'=> 0,
        'pending'    => 0,
        'delivered'  => 0
    ];
    
    foreach($orders as $o) {
        if ($o['status'] != 'cancelled') $stats['total_spent'] += $o['total_price'];
        if ($o['status'] == 'pending')   $stats['pending']++;
        if ($o['status'] == 'delivered') $stats['delivered']++;
    }
    
} catch (Exception $e) {
    $orders = [];
    $stats = ['total' => 0, 'total_spent' => 0, 'pending' => 0, 'delivered' => 0];
}

function getStatusConfig($status) {
    $configs = [
        'pending'   => ['label' => 'در انتظار',      'class' => 'bg-amber-50 text-amber-700 border-amber-200',   'icon' => 'fa-clock'],
        'processing'=> ['label' => 'در حال شستشو',   'class' => 'bg-blue-50 text-blue-700 border-blue-200',     'icon' => 'fa-wand-magic-sparkles'],
        'ready'     => ['label' => 'آماده تحویل',    'class' => 'bg-emerald-50 text-emerald-700 border-emerald-200','icon' => 'fa-bell-concierge'],
        'delivered' => ['label' => 'تحویل شده',      'class' => 'bg-zinc-100 text-zinc-700 border-zinc-200',    'icon' => 'fa-box-circle-check'],
        'cancelled' => ['label' => 'لغو شده',        'class' => 'bg-rose-50 text-rose-700 border-rose-200',     'icon' => 'fa-triangle-exclamation']
    ];
    return $configs[$status] ?? $configs['pending'];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>داشبورد کاربر | پاکشور</title>

    <!-- Tailwind -->
    <script src="https://cdn.tailwindcss.com"></script>

    <!-- Alpine -->
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>

    <!-- Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">

    <!-- Font -->
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">

    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        sans: ['Vazirmatn', 'system-ui', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50: '#eff6ff',
                            100: '#dbeafe',
                            500: '#3b82f6',
                            600: '#2563eb',
                            700: '#1d4ed8',
                        },
                        emerald: {
                            50: '#ecfdf5',
                            500: '#10b981',
                            600: '#059669',
                        }
                    },
                    boxShadow: {
                        soft: '0 18px 45px rgba(15,23,42,0.06)',
                        card: '0 10px 30px rgba(15,23,42,0.04)',
                    },
                    borderRadius: {
                        xl2: '1.25rem',
                    }
                }
            }
        }
    </script>

    <style>
        body { font-family: 'Vazirmatn', system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; }
        [x-cloak] { display: none !important; }
    </style>
</head>

<body class="bg-gradient-to-br from-slate-50 via-white to-emerald-50 text-zinc-800 min-h-screen" x-data="userDashboard()">

    <!-- پس‌زمینه دایره‌ای نرم -->
    <div class="fixed inset-0 -z-10 overflow-hidden pointer-events-none">
        <div class="absolute -left-24 -top-24 w-72 h-72 bg-primary-100 rounded-full blur-3xl opacity-70"></div>
        <div class="absolute -right-24 top-32 w-80 h-80 bg-emerald-100 rounded-full blur-3xl opacity-70"></div>
    </div>

    <!-- Header -->
    <header class="bg-white/80 backdrop-blur-xl border-b border-slate-200/70 sticky top-0 z-40">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 h-16 flex items-center justify-between">
            <!-- برند -->
            <a href="../../index.php" class="flex items-center gap-3">
                <div class="w-9 h-9 sm:w-10 sm:h-10 rounded-2xl bg-gradient-to-br from-primary-500 to-emerald-500 flex items-center justify-center text-white shadow-soft">
                    <i class="fa-solid fa-soap text-lg"></i>
                </div>
                <div class="flex flex-col">
                    <span class="font-extrabold text-base sm:text-lg text-slate-900 tracking-tight">پاکشور</span>
                    <span class="text-[11px] sm:text-xs text-slate-500">داشبورد مشتری • خشکشویی آنلاین</span>
                </div>
            </a>

            <!-- پروفایل / لاگ‌اوت -->
            <div class="flex items-center gap-4">
                <div class="hidden sm:flex flex-col items-end text-xs sm:text-sm">
                    <span class="font-medium text-slate-800">
                        <?php echo htmlspecialchars($_SESSION['full_name']); ?>
                    </span>
                    <span class="text-[11px] text-slate-500">کاربر عادی</span>
                </div>

                <div class="w-9 h-9 rounded-full bg-gradient-to-br from-slate-100 to-slate-200 flex items-center justify-center text-slate-600 shadow-sm">
                    <i class="fa-solid fa-user"></i>
                </div>

                <a href="../../logout.php"
                   class="inline-flex items-center gap-1.5 rounded-full border border-rose-200 bg-rose-50 px-3 py-1.5 text-[11px] sm:text-xs text-rose-600 hover:bg-rose-100 hover:border-rose-300 transition">
                    <i class="fa-solid fa-arrow-right-from-bracket text-xs"></i>
                    خروج
                </a>
            </div>
        </div>
    </header>

    <!-- Main -->
    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 sm:py-8 lg:py-10 space-y-6 sm:space-y-8">
        
        <!-- خوش‌آمدگویی -->
        <section class="hidden sm:flex items-center justify-between bg-white/80 backdrop-blur-lg border border-slate-200 rounded-3xl px-5 sm:px-7 py-4 shadow-soft">
            <div class="space-y-1">
                <h1 class="text-base sm:text-lg font-bold text-slate-900 flex items-center gap-2">
                    <span>سلام، <?php echo htmlspecialchars($_SESSION['full_name']); ?></span>
                    <span class="inline-flex items-center gap-1 rounded-full bg-emerald-50 border border-emerald-200 px-2 py-0.5 text-[11px] text-emerald-700">
                        <i class="fa-solid fa-sparkles text-[10px]"></i>
                        خوش آمدید
                    </span>
                </h1>
                <p class="text-xs sm:text-sm text-slate-500">
                    در این داشبورد می‌توانید سفارشات قبلی خود را ببینید، وضعیت آن‌ها را پیگیری کنید و جزئیات هر سفارش را بررسی کنید.
                </p>
            </div>

            <div class="hidden md:flex items-center gap-3">
                <div class="flex items-center gap-2 rounded-2xl bg-primary-50 border border-primary-100 px-3 py-2 text-[11px] text-primary-700">
                    <i class="fa-solid fa-circle-info text-primary-500"></i>
                    آخرین سفارش شما در این بخش به‌صورت زنده نمایش داده می‌شود.
                </div>
            </div>
        </section>

        <!-- کارت‌های آمار -->
        <section class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
            <div class="bg-white rounded-2xl border border-slate-200 shadow-card p-4 sm:p-5 flex flex-col justify-between">
                <div class="flex justify-between items-start mb-3">
                    <span class="text-xs sm:text-sm text-slate-500 font-medium">کل سفارشات</span>
                    <div class="w-9 h-9 rounded-xl bg-primary-50 text-primary-600 flex items-center justify-center">
                        <i class="fa-solid fa-box-archive"></i>
                    </div>
                </div>
                <div>
                    <div class="text-2xl sm:text-3xl font-black text-slate-900">
                        <?php echo number_format($stats['total']); ?>
                    </div>
                    <p class="text-[11px] sm:text-xs text-slate-400 mt-1">مجموع سفارش‌های ثبت‌شده</p>
                </div>
            </div>

            <div class="bg-white rounded-2xl border border-slate-200 shadow-card p-4 sm:p-5 flex flex-col justify-between">
                <div class="flex justify-between items-start mb-3">
                    <span class="text-xs sm:text-sm text-slate-500 font-medium">هزینه کل</span>
                    <div class="w-9 h-9 rounded-xl bg-emerald-50 text-emerald-600 flex items-center justify-center">
                        <i class="fa-solid fa-coins"></i>
                    </div>
                </div>
                <div>
                    <div class="text-2xl sm:text-3xl font-black text-slate-900">
                        <?php echo number_format($stats['total_spent']); ?>
                    </div>
                    <p class="text-[11px] sm:text-xs text-slate-400 mt-1">تومان، بدون سفارشات لغو شده</p>
                </div>
            </div>

            <div class="bg-white rounded-2xl border border-slate-200 shadow-card p-4 sm:p-5 flex flex-col justify-between">
                <div class="flex justify-between items-start mb-3">
                    <span class="text-xs sm:text-sm text-slate-500 font-medium">در انتظار بررسی / شستشو</span>
                    <div class="w-9 h-9 rounded-xl bg-amber-50 text-amber-600 flex items-center justify-center">
                        <i class="fa-solid fa-timer fa-clock"></i>
                    </div>
                </div>
                <div>
                    <div class="text-2xl sm:text-3xl font-black text-slate-900">
                        <?php echo number_format($stats['pending']); ?>
                    </div>
                    <p class="text-[11px] sm:text-xs text-slate-400 mt-1">سفارشی که هنوز تکمیل نشده‌اند</p>
                </div>
            </div>

            <div class="bg-white rounded-2xl border border-slate-200 shadow-card p-4 sm:p-5 flex flex-col justify-between">
                <div class="flex justify-between items-start mb-3">
                    <span class="text-xs sm:text-sm text-slate-500 font-medium">تحویل شده</span>
                    <div class="w-9 h-9 rounded-xl bg-sky-50 text-sky-600 flex items-center justify-center">
                        <i class="fa-solid fa-box-circle-check"></i>
                    </div>
                </div>
                <div>
                    <div class="text-2xl sm:text-3xl font-black text-slate-900">
                        <?php echo number_format($stats['delivered']); ?>
                    </div>
                    <p class="text-[11px] sm:text-xs text-slate-400 mt-1">سفارش‌های به‌طور کامل تحویل‌شده</p>
                </div>
            </div>
        </section>

        <?php if(!empty($orders)): 
            $latestOrder  = $orders[0];
            $latestStatus = getStatusConfig($latestOrder['status']);

            $steps = [
                'pending'    => ['label' => 'ثبت سفارش',   'icon' => 'fa-list-check'],
                'processing' => ['label' => 'در حال شستشو','icon' => 'fa-wand-magic-sparkles'],
                'ready'      => ['label' => 'آماده تحویل', 'icon' => 'fa-bell-concierge'],
                'delivered'  => ['label' => 'تحویل شده',   'icon' => 'fa-box-circle-check'],
            ];
            $stepKeys    = array_keys($steps);
            $currentStep = array_search($latestOrder['status'], $stepKeys);
            if($currentStep === false) $currentStep = 0;
        ?>
        <!-- جدید: کارت تایم‌لاین آخرین سفارش -->
        <section class="bg-white/90 border border-slate-200 rounded-3xl shadow-soft p-5 sm:p-6 lg:p-7 space-y-4 sm:space-y-5">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-3">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-2xl bg-gradient-to-br from-primary-500 to-emerald-500 text-white flex items-center justify-center shadow-md">
                        <i class="fa-solid fa-receipt text-lg"></i>
                    </div>
                    <div>
                        <h3 class="font-bold text-slate-900 text-sm sm:text-base">آخرین سفارش شما</h3>
                        <p class="text-[11px] sm:text-xs text-slate-500">
                            وضعیت این سفارش به‌صورت مرحله‌ای در نوار زیر نمایش داده می‌شود.
                        </p>
                    </div>
                </div>

                <div class="flex flex-wrap items-center gap-2 text-[11px] sm:text-xs">
                    <span class="inline-flex items-center gap-1 rounded-full bg-slate-50 border border-slate-200 px-2.5 py-1 font-mono text-slate-700">
                        <i class="fa-solid fa-barcode text-slate-500"></i>
                        <?php echo $latestOrder['tracking_code'] ?? '#'.$latestOrder['id']; ?>
                    </span>
                    <span class="inline-flex items-center gap-1 rounded-full <?php echo $latestStatus['class']; ?> border px-2.5 py-1">
                        <i class="fa-solid <?php echo $latestStatus['icon']; ?>"></i>
                        <?php echo $latestStatus['label']; ?>
                    </span>
                </div>
            </div>

            <!-- Steps -->
            <div class="relative mt-2">
                <div class="flex items-center justify-between mb-2">
                    <?php foreach($steps as $key => $step): 
                        $stepIndex = array_search($key, $stepKeys);
                        $isActive  = $stepIndex <= $currentStep;
                        $isCurrent = $stepIndex == $currentStep;
                    ?>
                    <div class="flex flex-col items-center relative z-10 flex-1">
                        <div data-step="<?php echo $key; ?>" class="w-9 h-9 sm:w-10 sm:h-10 rounded-full flex items-center justify-center mb-1.5 text-xs sm:text-sm transition-all
                            <?php echo $isActive ? 'bg-primary-600 text-white shadow-md' : 'bg-slate-100 text-slate-400 border border-slate-200'; ?>
                            <?php echo $isCurrent && $key == 'processing' ? ' ring-2 ring-primary-200' : ''; ?>">
                            <i class="fa-solid <?php echo $step['icon']; ?> <?php echo $isCurrent && $key == 'processing' ? 'fa-spin' : ''; ?>"></i>
                        </div>
                        <span data-step-label="<?php echo $key; ?>" class="text-[10px] sm:text-xs font-medium whitespace-nowrap
                            <?php echo $isActive ? 'text-primary-700' : 'text-slate-400'; ?>">
                            <?php echo $step['label']; ?>
                        </span>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- Progress Line -->
                <div class="absolute top-4 left-6 right-6 sm:left-10 sm:right-10 h-0.5 bg-slate-200 -z-0">
                    <div class="progress-line h-full bg-gradient-to-r from-primary-500 to-emerald-500 transition-all duration-500"
                         style="width: <?php echo ($currentStep / (count($steps) - 1)) * 100; ?>%">
                    </div>
                </div>
            </div>

            <div class="mt-3 p-3.5 sm:p-4 bg-slate-50 border border-slate-200 rounded-2xl flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2 text-xs sm:text-sm">
                <div class="flex items-center gap-2 text-slate-600">
                    <i class="fa-solid fa-sack-dollar text-emerald-500"></i>
                    <span>مبلغ:</span>
                    <span class="font-bold text-slate-900">
                        <?php echo number_format($latestOrder['total_price']); ?> تومان
                    </span>
                </div>
                <div class="flex items-center gap-2 text-slate-600">
                    <i class="fa-solid fa-calendar-days text-primary-500"></i>
                    <span>تاریخ ثبت:</span>
                    <span class="font-medium text-slate-900">
                        <?php echo $latestOrder['created_at_persian']; ?>
                    </span>
                </div>
                <?php if($latestOrder['delivery_date']): ?>
                <div class="flex items-center gap-2 text-slate-600">
                    <i class="fa-solid fa-truck text-emerald-500"></i>
                    <span>تاریخ تحویل:</span>
                    <span class="font-medium text-slate-900">
                        <?php echo $latestOrder['delivery_date']; ?>
                    </span>
                </div>
                <?php endif; ?>
            </div>
        </section>
        <?php else: ?>
        <!-- Empty state اگر سفارشی نیست -->
        <section class="bg-white/90 border border-dashed border-slate-300 rounded-3xl shadow-soft p-8 text-center space-y-3">
            <div class="w-14 h-14 mx-auto rounded-2xl bg-slate-50 text-slate-400 flex items-center justify-center border border-slate-200">
                <i class="fa-regular fa-folder-open text-2xl"></i>
            </div>
            <h3 class="text-sm sm:text-base font-bold text-slate-800">هنوز سفارشی ثبت نکرده‌اید</h3>
            <p class="text-xs sm:text-sm text-slate-500 max-w-md mx-auto">
                با ثبت اولین سفارش خشکشویی آنلاین خود در پاکشور، وضعیت آن را به صورت لحظه‌ای در همین داشبورد مشاهده خواهید کرد.
            </p>
            <a href="../../index.php"
               class="inline-flex items-center gap-2 mt-2 px-4 py-2.5 rounded-full bg-gradient-to-r from-primary-600 to-emerald-600 text-xs sm:text-sm text-white font-semibold shadow-md hover:from-primary-500 hover:to-emerald-500 transition">
                <i class="fa-solid fa-plus"></i>
                ثبت اولین سفارش
            </a>
        </section>
        <?php endif; ?>

        <!-- جدول سفارش‌ها -->
        <section class="bg-white/95 border border-slate-200 rounded-3xl shadow-soft overflow-hidden">
            <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3 px-5 sm:px-6 py-4 border-b border-slate-100">
                <div>
                    <h2 class="font-bold text-slate-900 text-sm sm:text-base">سفارشات من</h2>
                    <p class="text-[11px] sm:text-xs text-slate-500 mt-0.5">
                        لیست کامل سفارشات شما همراه با وضعیت، مبلغ و امکان مشاهده جزئیات هر سفارش.
                    </p>
                </div>
                <div class="inline-flex items-center gap-2 text-[11px] sm:text-xs text-slate-500 bg-slate-50 border border-slate-200 rounded-full px-3 py-1.5">
                    <i class="fa-solid fa-filter-circle-dollar text-slate-400"></i>
                    <span>برای مشاهده جزئیات هر سفارش، روی «جزئیات» کلیک کنید.</span>
                </div>
            </div>

            <div class="overflow-x-auto">
                <table class="w-full text-right text-sm">
                    <thead class="bg-slate-50/80 border-b border-slate-200 text-xs text-slate-500">
                        <tr>
                            <th class="px-6 py-3.5 font-semibold">کد پیگیری</th>
                            <th class="px-6 py-3.5 font-semibold">تاریخ</th>
                            <th class="px-6 py-3.5 font-semibold">مبلغ</th>
                            <th class="px-6 py-3.5 font-semibold">وضعیت</th>
                            <th class="px-6 py-3.5 font-semibold text-center">عملیات</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-slate-100">
                        <?php if(empty($orders)): ?>
                        <tr>
                            <td colspan="5" class="px-6 py-10 text-center text-slate-500 text-sm">
                                سفارشی یافت نشد.
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach($orders as $o): 
                            $status = getStatusConfig($o['status']);
                        ?>
                        <tr class="hover:bg-slate-50/70 transition-colors">
                            <td class="px-6 py-3.5 align-middle">
                                <div class="flex flex-col gap-0.5">
                                    <span class="font-mono font-bold text-slate-800 text-sm">
                                        <?php echo $o['tracking_code'] ?? $o['id']; ?>
                                    </span>
                                    <span class="text-[11px] text-slate-400">
                                        #<?php echo $o['id']; ?>
                                    </span>
                                </div>
                            </td>
                            <td class="px-6 py-3.5 align-middle text-xs sm:text-sm text-slate-600">
                                <div class="flex flex-col gap-0.5">
                                    <span><?php echo $o['created_at_persian']; ?></span>
                                </div>
                            </td>
                            <td class="px-6 py-3.5 align-middle">
                                <div class="flex flex-col items-start">
                                    <span class="font-bold text-slate-900 text-sm">
                                        <?php echo number_format($o['total_price']); ?> <span class="text-[11px] text-slate-500">تومان</span>
                                    </span>
                                </div>
                            </td>
                            <td class="px-6 py-3.5 align-middle">
                                <span class="inline-flex items-center gap-1.5 px-2.5 py-1.5 rounded-full text-[11px] sm:text-xs font-medium border <?php echo $status['class']; ?>">
                                    <i class="fa-solid <?php echo $status['icon']; ?>"></i>
                                    <?php echo $status['label']; ?>
                                </span>
                            </td>
                            <td class="px-6 py-3.5 align-middle text-center">
                                <button
                                    @click="openDetail(<?php echo htmlspecialchars(json_encode($o)); ?>)"
                                    class="inline-flex items-center gap-1.5 rounded-full border border-primary-200 bg-primary-50 px-3 py-1.5 text-[11px] sm:text-xs text-primary-700 hover:bg-primary-100 hover:border-primary-300 transition">
                                    <i class="fa-solid fa-eye"></i>
                                    جزئیات
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </section>
    </main>

    <!-- Modal جزئیات سفارش -->
    <div x-show="detailOpen" x-cloak class="fixed inset-0 z-50 flex items-center justify-center px-4 py-6 sm:py-10">
        <div class="fixed inset-0 bg-black/40 backdrop-blur-sm" @click="detailOpen = false"></div>

        <div class="relative bg-white rounded-3xl shadow-2xl max-w-2xl w-full p-5 sm:p-6 lg:p-7 z-10">
            <div class="flex justify-between items-start mb-5 sm:mb-6">
                <div>
                    <div class="inline-flex items-center gap-2 rounded-full bg-slate-50 border border-slate-200 px-3 py-1 mb-2">
                        <i class="fa-solid fa-receipt text-slate-500 text-xs"></i>
                        <span class="text-[11px] text-slate-600">جزئیات سفارش</span>
                    </div>
                    <h3 class="text-base sm:text-lg font-bold text-slate-900 mb-1">
                        کد پیگیری:
                        <span class="font-mono text-primary-600" x-text="activeOrder?.tracking_code || ('#' + activeOrder?.id)"></span>
                    </h3>
                    <p class="text-xs sm:text-sm text-slate-500">
                        جزئیات خدمات، مبلغ و زمان ثبت سفارش در این بخش نمایش داده می‌شود.
                    </p>
                </div>
                <button @click="detailOpen = false"
                        class="w-8 h-8 rounded-full flex items-center justify-center bg-slate-100 text-slate-500 hover:bg-slate-200 hover:text-slate-700 transition">
                    <i class="fa-solid fa-xmark text-lg"></i>
                </button>
            </div>

            <div class="space-y-4 sm:space-y-5">
                <!-- اطلاعات کلی -->
                <div class="bg-slate-50 border border-slate-200 rounded-2xl p-3.5 sm:p-4">
                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 text-xs sm:text-sm">
                        <div class="flex flex-col gap-1">
                            <span class="text-slate-500">تاریخ ثبت سفارش</span>
                            <span class="font-medium text-slate-900" x-text="activeOrder?.created_at_persian || formatDate(activeOrder?.created_at)"></span>
                        </div>
                        <div class="flex flex-col gap-1">
                            <span class="text-slate-500">مبلغ کل</span>
                            <span class="font-bold text-emerald-600" x-text="formatPrice(activeOrder?.total_price)"></span>
                        </div>
                        <div class="flex flex-col gap-1" x-show="activeOrder?.delivery_date">
                            <span class="text-slate-500">تاریخ تحویل</span>
                            <span class="font-medium text-slate-900" x-text="activeOrder?.delivery_date"></span>
                        </div>
                    </div>
                </div>

                <!-- جدول آیتم‌ها -->
                <div class="border border-slate-200 rounded-2xl overflow-hidden">
                    <div class="px-4 py-3 bg-slate-50 border-b border-slate-200 flex items-center justify-between text-xs sm:text-sm text-slate-600">
                        <span class="flex items-center gap-2">
                            <i class="fa-solid fa-list-check text-slate-500"></i>
                            لیست خدمات این سفارش
                        </span>
                        <span class="text-[11px]" x-show="activeItems?.length" x-text="activeItems.length + ' مورد'"></span>
                    </div>
                    <div class="overflow-x-auto">
                        <table class="w-full text-xs sm:text-sm">
                            <thead class="bg-slate-50">
                                <tr>
                                    <th class="px-4 py-2 text-right font-semibold text-slate-500">خدمت</th>
                                    <th class="px-4 py-2 text-center font-semibold text-slate-500">تعداد</th>
                                    <th class="px-4 py-2 text-left font-semibold text-slate-500">مبلغ</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-slate-100" x-show="activeItems?.length">
                                <template x-for="item in activeItems" :key="item.id">
                                    <tr>
                                        <td class="px-4 py-3 text-slate-800" x-text="item.name"></td>
                                        <td class="px-4 py-3 text-center text-slate-600" x-text="item.qty"></td>
                                        <td class="px-4 py-3 text-left font-medium text-slate-900" x-text="formatPrice(item.price * item.qty)"></td>
                                    </tr>
                                </template>
                            </tbody>
                            <tbody x-show="!activeItems?.length">
                                <tr>
                                    <td colspan="3" class="px-4 py-5 text-center text-slate-400 text-xs sm:text-sm">
                                        جزئیات خدمات برای این سفارش ثبت نشده است.
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- یادداشت (در صورت وجود) -->
                <div x-show="activeOrder?.notes" class="bg-sky-50 border border-sky-200 rounded-2xl p-3.5 sm:p-4">
                    <div class="flex items-center gap-2 text-xs sm:text-sm text-sky-700 font-semibold mb-1.5">
                        <i class="fa-solid fa-note-sticky text-sky-500 text-sm"></i>
                        یادداشت سفارش
                    </div>
                    <p class="text-xs sm:text-sm text-sky-800 leading-relaxed" x-text="activeOrder?.notes"></p>
                </div>
            </div>

            <div class="mt-5 sm:mt-6 flex justify-end gap-2">
                <button @click="detailOpen = false"
                        class="px-4 py-2.5 rounded-xl border border-slate-200 bg-white text-xs sm:text-sm text-slate-700 hover:bg-slate-50 transition">
                    بستن
                </button>
            </div>
        </div>
    </div>

    <script>
        function userDashboard() {
            return {
                detailOpen: false,
                activeOrder: null,
                activeItems: [],
                latestOrderStatus: '<?php echo !empty($orders) ? $orders[0]['status'] : ''; ?>',
                latestOrderId: <?php echo !empty($orders) ? $orders[0]['id'] : 'null'; ?>,

                init() {
                    // Check for status updates every 5 seconds
                    if (this.latestOrderId) {
                        setInterval(() => {
                            this.checkStatusUpdate();
                        }, 5000);
                    }
                },

                async checkStatusUpdate() {
                    try {
                        const response = await fetch(`../../api/orders.php?id=${this.latestOrderId}`);
                        const order = await response.json();
                        
                        if (order && order.status !== this.latestOrderStatus) {
                            this.latestOrderStatus = order.status;
                            this.updateProgressBar(order.status);
                        }
                    } catch (error) {
                        console.log('Status check failed');
                    }
                },

                updateProgressBar(newStatus) {
                    const steps = ['pending', 'processing', 'ready', 'delivered'];
                    const currentStep = steps.indexOf(newStatus);
                    
                    // Update progress line
                    const progressLine = document.querySelector('.progress-line');
                    if (progressLine) {
                        progressLine.style.width = `${(currentStep / (steps.length - 1)) * 100}%`;
                    }
                    
                    // Update step indicators
                    steps.forEach((step, index) => {
                        const stepElement = document.querySelector(`[data-step="${step}"]`);
                        const labelElement = document.querySelector(`[data-step-label="${step}"]`);
                        const isActive = index <= currentStep;
                        const isCurrent = index === currentStep;
                        
                        if (stepElement) {
                            stepElement.className = `w-9 h-9 sm:w-10 sm:h-10 rounded-full flex items-center justify-center mb-1.5 text-xs sm:text-sm transition-all ${
                                isActive ? 'bg-primary-600 text-white shadow-md' : 'bg-slate-100 text-slate-400 border border-slate-200'
                            } ${isCurrent && step === 'processing' ? 'ring-2 ring-primary-200' : ''}`;
                            
                            const icon = stepElement.querySelector('i');
                            if (icon) {
                                if (step === 'processing') {
                                    icon.className = `fa-solid fa-wand-magic-sparkles ${isCurrent ? 'fa-spin' : ''}`;
                                } else {
                                    const iconMap = {
                                        'pending': 'fa-list-check',
                                        'ready': 'fa-bell-concierge', 
                                        'delivered': 'fa-box-circle-check'
                                    };
                                    icon.className = `fa-solid ${iconMap[step] || 'fa-question'}`;
                                }
                            }
                        }
                        
                        if (labelElement) {
                            labelElement.className = `text-[10px] sm:text-xs font-medium whitespace-nowrap ${
                                isActive ? 'text-primary-700' : 'text-slate-400'
                            }`;
                        }
                    });
                },

                openDetail(order) {
                    this.activeOrder = order;
                    try {
                        this.activeItems = typeof order.items_json === 'string'
                            ? JSON.parse(order.items_json)
                            : (order.items_json || []);
                    } catch (e) {
                        this.activeItems = [];
                    }
                    this.detailOpen = true;
                },

                formatPrice(price) {
                    if (!price && price !== 0) return '-';
                    return new Intl.NumberFormat('fa-IR').format(price) + ' تومان';
                },

                formatDate(timestamp) {
                    if (!timestamp) return '-';
                    if (typeof timestamp === 'string' && isNaN(parseInt(timestamp))) {
                        return timestamp;
                    }
                    const ts = parseInt(timestamp);
                    const date = new Date(ts > 10000000000 ? ts : ts * 1000);
                    return date.toLocaleDateString('fa-IR') + ' ' +
                        date.toLocaleTimeString('fa-IR', { hour: '2-digit', minute: '2-digit' });
                }
            }
        }
    </script>
</body>
</html>
