<?php
session_start();
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

require_once '../../api/db.php';
require_once '../../api/persian_date.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../login.php");
    exit();
}

// Fetch Services
$services = [];
try {
    $stmt = $conn->query("SELECT * FROM services ORDER BY id");
    $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($services)) {
        $conn->exec("INSERT INTO services (name, price, icon) VALUES 
            ('پیراهن مردانه', 50000, 'fa-shirt'),
            ('شلوار', 60000, 'fa-layer-group'),
            ('کت و شلوار', 180000, 'fa-user-tie'),
            ('کاپشن / پالتو', 150000, 'fa-snowflake'),
            ('پتو یک نفره', 120000, 'fa-bed'),
            ('پرده (هر پنل)', 80000, 'fa-dungeon'),
            ('مانتو', 70000, 'fa-vest'),
            ('چادر', 65000, 'fa-person-dress'),
            ('کیف و کفش', 90000, 'fa-shoe-prints')
        ");
        $stmt = $conn->query("SELECT * FROM services ORDER BY id");
        $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
} catch (PDOException $e) {
    die("خطا: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ثبت سفارش جدید</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../../css/style.css">
</head>
<body class="bg-gray-100 font-vazir">

<div class="flex h-screen overflow-hidden">
    
    <!-- Sidebar -->
    <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl">
        <div class="h-16 flex items-center justify-center border-b border-slate-700 font-bold text-xl">
            پنل مدیریت
        </div>
        <nav class="flex-1 p-4 space-y-2">
            <a href="admin.php" class="flex items-center px-4 py-3 text-slate-300 hover:bg-slate-800 rounded-xl transition">
                <i class="fa-solid fa-list-check ml-3 w-6"></i> سفارشات
            </a>
            <a href="order_create.php" class="flex items-center px-4 py-3 bg-blue-600 text-white rounded-xl">
                <i class="fa-solid fa-cart-plus ml-3 w-6"></i> ثبت سفارش
            </a>
        </nav>
        <div class="p-4 border-t border-slate-700">
            <a href="../../logout.php" class="flex items-center justify-center px-4 py-2 bg-red-600/20 hover:bg-red-600 text-red-100 rounded-xl transition">
                <i class="fa-solid fa-power-off ml-2"></i> خروج
            </a>
        </div>
    </aside>

    <!-- Main -->
    <main class="flex-1 flex flex-col overflow-hidden">
        
        <!-- Header -->
        <header class="bg-white px-6 py-4 shadow-sm flex justify-between items-center">
            <h1 class="text-xl font-bold text-slate-800">
                <i class="fa-solid fa-plus-circle text-blue-600"></i> ثبت سفارش جدید
            </h1>
            <a href="admin.php" class="md:hidden text-slate-500"><i class="fa-solid fa-arrow-left"></i></a>
        </header>

        <div class="flex-1 overflow-hidden flex flex-col lg:flex-row">
            
            <!-- Services List -->
            <div class="flex-1 overflow-y-auto p-4 md:p-6 pb-24 lg:pb-6">
                
                <!-- Customer Info -->
                <div class="bg-white rounded-2xl shadow-sm border p-6 mb-6">
                    <h3 class="font-bold text-slate-700 mb-4 border-b pb-2">اطلاعات مشتری</h3>
                    
                    <!-- Tracking & Date -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-bold text-slate-600 mb-2">شماره پیگیری</label>
                            <div class="flex items-center gap-2">
                                <input type="text" id="tracking-code" readonly class="flex-1 px-4 py-3 rounded-xl border bg-blue-50 text-blue-700 font-bold text-lg" value="<?php 
                                    list($jy, $jm, $jd) = gregorian_to_jalali(date('Y'), date('m'), date('d'));
                                    echo str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT) . '-' . $jy; 
                                ?>">
                                <button type="button" onclick="generateTrackingCode()" class="px-4 py-3 bg-blue-600 text-white rounded-xl hover:bg-blue-700 transition" title="تولید کد جدید">
                                    <i class="fa-solid fa-rotate"></i>
                                </button>
                            </div>
                            <p class="text-xs text-slate-500 mt-1"><i class="fa-solid fa-info-circle"></i> این کد در فاکتور نمایش داده می‌شود</p>
                        </div>
                        <div>
                            <label class="block text-sm font-bold text-slate-600 mb-2">تاریخ</label>
                            <input type="text" readonly class="w-full px-4 py-3 rounded-xl border bg-slate-50 text-slate-700" value="<?php echo toPersianNum(getPersianDate()); ?>">
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-bold text-slate-600 mb-2">نام و نام خانوادگی <span class="text-red-500">*</span></label>
                            <input type="text" id="cust-name" class="w-full px-4 py-3 rounded-xl border focus:border-blue-500 focus:ring-2 focus:ring-blue-100 outline-none" placeholder="علی رضایی">
                        </div>
                        <div>
                            <label class="block text-sm font-bold text-slate-600 mb-2">شماره تماس <span class="text-red-500">*</span></label>
                            <input type="tel" id="cust-phone" class="w-full px-4 py-3 rounded-xl border focus:border-blue-500 focus:ring-2 focus:ring-blue-100 outline-none" placeholder="09123456789">
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-bold text-slate-600 mb-2">آدرس</label>
                        <textarea id="cust-address" class="w-full px-4 py-3 rounded-xl border focus:border-blue-500 focus:ring-2 focus:ring-blue-100 outline-none resize-none h-20" placeholder="آدرس دقیق..."></textarea>
                    </div>
                </div>

                <!-- Services -->
                <div class="mb-4 flex justify-between items-center">
                    <h3 class="font-bold text-slate-700">لیست خدمات (<?php echo count($services); ?> مورد)</h3>
                    <span class="text-xs text-slate-500 bg-slate-100 px-3 py-1 rounded-full">
                        <i class="fa-solid fa-info-circle"></i> روی هر خدمت کلیک کنید
                    </span>
                </div>

                <?php if (empty($services)): ?>
                    <div class="p-8 text-center bg-red-50 text-red-600 rounded-2xl">
                        <i class="fa-solid fa-exclamation-triangle text-3xl mb-2"></i>
                        <p>خدماتی یافت نشد!</p>
                    </div>
                <?php else: ?>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                        <?php foreach($services as $s): ?>
                        <div class="bg-white p-5 rounded-2xl border-2 border-slate-200 hover:border-blue-500 hover:shadow-lg transition cursor-pointer relative group"
                             onclick="addToCart(<?php echo $s['id']; ?>, '<?php echo htmlspecialchars($s['name'], ENT_QUOTES); ?>', <?php echo $s['price']; ?>)">
                            
                            <!-- Badge -->
                            <div id="badge-<?php echo $s['id']; ?>" class="hidden absolute -top-2 -left-2 bg-blue-600 text-white text-xs font-bold min-w-[24px] h-6 px-2 rounded-full items-center justify-center shadow-lg border-2 border-white">0</div>

                            <!-- Icon & Info -->
                            <div class="flex items-start gap-3 mb-3">
                                <div class="w-14 h-14 rounded-xl bg-blue-50 text-blue-600 flex items-center justify-center text-2xl group-hover:scale-110 transition">
                                    <i class="fa-solid <?php echo $s['icon']; ?>"></i>
                                </div>
                                <div class="flex-1">
                                    <h4 class="font-bold text-slate-800 mb-1"><?php echo $s['name']; ?></h4>
                                    <div class="text-blue-600 font-bold text-sm">
                                        <i class="fa-solid fa-tag text-xs"></i> <?php echo number_format($s['price']); ?> تومان
                                    </div>
                                </div>
                            </div>

                            <!-- Details -->
                            <div class="bg-slate-50 rounded-lg p-2 mb-3 text-xs text-slate-600">
                                <i class="fa-solid fa-clock"></i> زمان تحویل: 24-48 ساعت
                            </div>

                            <!-- Button -->
                            <div class="bg-blue-600 text-white py-2 rounded-xl text-center font-bold text-sm group-hover:bg-blue-700 transition">
                                <i class="fa-solid fa-plus"></i> افزودن به سبد
                            </div>

                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

            </div>

            <!-- Cart Sidebar -->
            <div class="fixed lg:relative bottom-0 left-0 right-0 lg:w-96 bg-white border-t lg:border-r lg:border-t-0 flex flex-col shadow-2xl lg:shadow-xl z-40 max-h-[70vh] lg:max-h-full">
                
                <div class="p-4 lg:p-5 border-b bg-slate-50 flex justify-between items-center lg:block">
                    <h3 class="font-bold text-slate-800">
                        <i class="fa-solid fa-basket-shopping text-blue-600"></i> سبد سفارش
                    </h3>
                    <button onclick="toggleCart()" class="lg:hidden text-slate-500 hover:text-slate-700">
                        <i class="fa-solid fa-chevron-down" id="cart-toggle-icon"></i>
                    </button>
                </div>

                <div id="cart-container" class="flex-1 overflow-y-auto p-3 lg:p-4 space-y-3">
                    <div class="text-center py-10 text-slate-400">
                        <i class="fa-solid fa-cart-arrow-down text-4xl mb-2 opacity-50"></i>
                        <p class="text-sm">سبد خالی است</p>
                    </div>
                </div>

                <div class="p-4 lg:p-5 border-t bg-white" id="cart-footer">
                    <div class="mb-4">
                        <label class="text-xs font-bold text-slate-500 mb-1 block">توضیحات (اختیاری)</label>
                        <textarea id="order-notes" class="w-full h-16 rounded-lg border p-2 text-sm focus:border-blue-500 outline-none resize-none" placeholder="یادداشت..."></textarea>
                    </div>
                    
                    <div class="flex justify-between items-center mb-4 text-lg font-bold">
                        <span>جمع کل:</span>
                        <span id="total-price" class="text-blue-600">0 تومان</span>
                    </div>

                    <button onclick="submitOrder()" id="btn-submit" disabled class="w-full py-3 lg:py-4 rounded-xl bg-blue-600 text-white font-bold shadow-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition">
                        ثبت نهایی سفارش
                    </button>
                </div>

                <!-- Mobile Cart Summary -->
                <div class="lg:hidden p-3 bg-white border-t flex items-center justify-between">
                    <div class="flex items-center gap-2">
                        <span class="text-sm text-slate-600">جمع کل:</span>
                        <span id="total-price-mobile" class="text-lg font-bold text-blue-600">0 تومان</span>
                    </div>
                    <button onclick="toggleCart()" class="bg-blue-600 text-white px-4 py-2 rounded-lg font-bold text-sm">
                        <i class="fa-solid fa-shopping-cart ml-1"></i> مشاهده سبد
                    </button>
                </div>

            </div>

        </div>
    </main>
</div>

<div id="toast-container" class="fixed top-4 left-4 z-50 space-y-2"></div>

<script src="../../js/utils.js"></script>
<script>
const cart = {};
let cartOpen = false;

function toggleCart() {
    cartOpen = !cartOpen;
    const container = document.getElementById('cart-container');
    const footer = document.getElementById('cart-footer');
    const icon = document.getElementById('cart-toggle-icon');
    
    if (cartOpen) {
        container.classList.remove('hidden');
        footer.classList.remove('hidden');
        icon.classList.remove('fa-chevron-down');
        icon.classList.add('fa-chevron-up');
    } else {
        container.classList.add('hidden');
        footer.classList.add('hidden');
        icon.classList.remove('fa-chevron-up');
        icon.classList.add('fa-chevron-down');
    }
}

function generateTrackingCode() {
    const persianYear = new Date().getFullYear() - 621;
    const random = String(Math.floor(Math.random() * 900000) + 100000);
    document.getElementById('tracking-code').value = `${random}-${persianYear}`;
}

function addToCart(id, name, price) {
    if (!cart[id]) {
        cart[id] = { id, name, price, qty: 0 };
    }
    cart[id].qty++;
    updateUI();
}

function removeFromCart(id) {
    if (cart[id]) {
        cart[id].qty--;
        if (cart[id].qty <= 0) delete cart[id];
    }
    updateUI();
}

function updateUI() {
    const container = document.getElementById('cart-container');
    const totalEl = document.getElementById('total-price');
    const submitBtn = document.getElementById('btn-submit');
    
    let total = 0;
    let html = '';
    let hasItems = false;

    document.querySelectorAll('[id^="badge-"]').forEach(el => {
        el.classList.add('hidden');
        el.classList.remove('flex');
    });

    for (const id in cart) {
        const item = cart[id];
        total += item.price * item.qty;
        hasItems = true;

        const badge = document.getElementById(`badge-${id}`);
        if (badge) {
            badge.innerText = item.qty;
            badge.classList.remove('hidden');
            badge.classList.add('flex');
        }

        html += `
            <div class="flex items-center justify-between bg-slate-50 p-3 rounded-xl border">
                <div>
                    <p class="font-bold text-sm text-slate-700">${item.name}</p>
                    <p class="text-xs text-slate-400">${(item.price * item.qty).toLocaleString()} تومان</p>
                </div>
                <div class="flex items-center gap-2 bg-white rounded-lg px-1 py-1 border">
                    <button onclick="addToCart(${item.id}, '${item.name}', ${item.price})" class="w-6 h-6 flex items-center justify-center text-green-600 hover:bg-green-50 rounded"><i class="fa-solid fa-plus text-xs"></i></button>
                    <span class="text-sm font-bold w-4 text-center">${item.qty}</span>
                    <button onclick="removeFromCart(${item.id})" class="w-6 h-6 flex items-center justify-center text-red-500 hover:bg-red-50 rounded"><i class="fa-solid fa-minus text-xs"></i></button>
                </div>
            </div>
        `;
    }

    if (!hasItems) {
        container.innerHTML = `
            <div class="text-center py-10 text-slate-400">
                <i class="fa-solid fa-cart-arrow-down text-4xl mb-2 opacity-50"></i>
                <p class="text-sm">سبد خالی است</p>
            </div>
        `;
        submitBtn.disabled = true;
        totalEl.innerText = '0 تومان';
        document.getElementById('total-price-mobile').innerText = '0 تومان';
    } else {
        container.innerHTML = html;
        submitBtn.disabled = false;
        totalEl.innerText = total.toLocaleString() + ' تومان';
        document.getElementById('total-price-mobile').innerText = total.toLocaleString() + ' تومان';
    }
}

async function submitOrder() {
    const name = document.getElementById('cust-name').value.trim();
    const phone = document.getElementById('cust-phone').value.trim();
    const address = document.getElementById('cust-address').value.trim();
    const notes = document.getElementById('order-notes').value.trim();

    if (name.length < 3) {
        showToast('لطفاً نام مشتری را وارد کنید', 'error');
        return;
    }
    if (phone.length < 10) {
        showToast('شماره موبایل معتبر نیست', 'error');
        return;
    }

    const items = [];
    let total = 0;
    for (const id in cart) {
        items.push(cart[id]);
        total += cart[id].price * cart[id].qty;
    }

    const trackingCode = document.getElementById('tracking-code').value;

    const payload = {
        customer_name: name,
        customer_phone: phone,
        customer_address: address,
        tracking_code: trackingCode,
        items_json: JSON.stringify(items),
        total_price: total,
        notes: notes,
        status: 'pending'
    };

    const btn = document.getElementById('btn-submit');
    btn.disabled = true;
    btn.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> در حال ثبت...';

    try {
        const response = await apiCall('../../api/orders.php', 'POST', payload);
        
        if (response && response.id) {
            showToast(`سفارش ثبت شد (کد: ${response.id})`, 'success');
            setTimeout(() => window.location.href = 'admin.php', 1500);
        } else {
            showToast('خطا در ثبت سفارش', 'error');
            btn.disabled = false;
            btn.innerHTML = 'ثبت نهایی سفارش';
        }
    } catch (err) {
        showToast('خطای ارتباط با سرور', 'error');
        btn.disabled = false;
        btn.innerHTML = 'ثبت نهایی سفارش';
    }
}
</script>

</body>
</html>
